# System Architecture & Flow Diagrams

## 🏗️ System Architecture

```
┌─────────────────────────────────────────────────────────────────┐
│                         USER BROWSER                             │
├─────────────────────────────────────────────────────────────────┤
│                                                                  │
│  ┌──────────────────┐  ┌──────────────────┐                    │
│  │   shop.php       │  │    cart.php      │                    │
│  │  (Shopping Page) │  │ (Cart Display)   │                    │
│  │                  │  │                  │                    │
│  │ • Product list   │  │ • Cart items     │                    │
│  │ • Add to Cart    │  │ • Remove item    │                    │
│  │ • Toast notify   │  │ • Total price    │                    │
│  └──────────────────┘  └──────────────────┘                    │
│          │                      │                               │
│          │      AJAX POST       │                               │
│          │     AJAX GET         │                               │
│          └──────────┬───────────┘                               │
│                     │                                            │
└─────────────────────┼────────────────────────────────────────────┘
                      │
            ┌─────────▼────────┐
            │   PHP Backend    │
            └──────────────────┘
                      │
        ┌─────────────┴─────────────┐
        │                           │
    ┌───▼────────┐         ┌────────▼──┐
    │ cart-api   │         │  Session  │
    │ .php       │         │  Management
    │            │         │            │
    │ • add      │         │ • Auth    │
    │ • remove   │         │ • User ID │
    │ • get      │         │ • Role    │
    │ • count    │         └────────────┘
    └────┬───────┘
         │
         │ SQL
         │
    ┌────▼──────────┐
    │    MySQL      │
    │   Database    │
    │               │
    │  ┌──────────┐ │
    │  │ users    │ │
    │  │ table    │ │
    │  └──────────┘ │
    │               │
    │  ┌──────────┐ │
    │  │  cart    │ │
    │  │  table   │ │
    │  └──────────┘ │
    └───────────────┘
```

---

## 🔄 Add to Cart Flow

```
┌─────────────────────────────────────────────────────────────────┐
│ USER CLICKS "ADD TO CART" BUTTON                                │
└────────────────────┬────────────────────────────────────────────┘
                     │
                     ▼
        ┌────────────────────────┐
        │  JavaScript captures:  │
        │  • product_id          │
        │  • product_name        │
        │  • price               │
        │  • image               │
        │  • quantity            │
        └────────┬───────────────┘
                 │
                 ▼
      ┌──────────────────────────┐
      │ AJAX POST Request Sent   │
      │ to cart-api.php?action=add
      └────────┬─────────────────┘
               │
               ▼
    ┌────────────────────────────────┐
    │    PHP Backend (cart-api.php)  │
    │                                │
    │ 1. Check session exists        │
    │    ✓ $_SESSION['user_id'] set? │
    │                                │
    │ 2. Verify user is customer     │
    │    ✓ $_SESSION['user_role'] == 'customer'?
    │                                │
    │ 3. Check if product in cart    │
    │    ✓ SELECT FROM cart          │
    │                                │
    │ 4. If exists → UPDATE qty      │
    │    UPDATE cart SET qty = qty+1 │
    │                                │
    │ 5. If not exists → INSERT      │
    │    INSERT INTO cart VALUES...  │
    │                                │
    │ 6. Get new cart count          │
    │    SELECT COUNT(*) FROM cart   │
    │                                │
    │ 7. Return JSON response        │
    │    {success: true, count: 5}   │
    └────────┬─────────────────────┘
             │
             ▼
   ┌──────────────────────────┐
   │ JavaScript receives:     │
   │ {                        │
   │   success: true,         │
   │   message: "Added!",     │
   │   cart_count: 5          │
   │ }                        │
   └────────┬─────────────────┘
            │
            ├─ Show green toast
            │  "Product added to cart"
            │
            ├─ Update cart count (5)
            │
            ├─ Reset quantity input to 1
            │
            └─ NO PAGE RELOAD ✅
```

---

## 🗑️ Remove from Cart Flow

```
┌─────────────────────────────────────────────────────────────────┐
│ USER CLICKS "REMOVE" BUTTON ON CART.PHP                         │
└────────────────────┬────────────────────────────────────────────┘
                     │
                     ▼
        ┌────────────────────────┐
        │  JavaScript captures:  │
        │  • cart_id (primary key)
        │                        │
        │  Button shows loading  │
        │  state (disabled)      │
        └────────┬───────────────┘
                 │
                 ▼
      ┌──────────────────────────┐
      │ AJAX POST Request Sent   │
      │ to cart-api.php?action=remove
      └────────┬─────────────────┘
               │
               ▼
    ┌────────────────────────────────┐
    │    PHP Backend (cart-api.php)  │
    │                                │
    │ 1. Check session exists        │
    │    ✓ $_SESSION['user_id'] set? │
    │                                │
    │ 2. Verify user is customer     │
    │    ✓ $_SESSION['user_role']    │
    │                                │
    │ 3. Verify ownership            │
    │    ✓ SELECT FROM cart          │
    │    ✓ WHERE id=? AND user_id=?  │
    │    ✓ Prevent unauthorized del  │
    │                                │
    │ 4. Delete cart item            │
    │    DELETE FROM cart WHERE id   │
    │                                │
    │ 5. Get new cart count          │
    │    SELECT COUNT(*) FROM cart   │
    │                                │
    │ 6. Return JSON response        │
    │    {success: true, count: 4}   │
    └────────┬─────────────────────┘
             │
             ▼
   ┌──────────────────────────┐
   │ JavaScript receives:     │
   │ {                        │
   │   success: true,         │
   │   message: "Removed!",   │
   │   cart_count: 4          │
   │ }                        │
   └────────┬─────────────────┘
            │
            ├─ Show green toast
            │  "Product removed"
            │
            ├─ Reload cart display
            │  (calls action=get)
            │
            ├─ Update cart UI
            │
            └─ NO PAGE REFRESH ✅
```

---

## 🔐 Authentication & Authorization Flow

```
┌──────────────────────────────────────┐
│ USER VISITS shop.php                 │
└────────────┬───────────────────────┘
             │
             ▼
    ┌─────────────────────┐
    │ Check: Is user      │
    │ logged in?          │
    │                     │
    │ isset($_SESSION[    │
    │ 'user_id'])?        │
    └────┬─────────────┬──┘
         │ NO          │ YES
         │             │
    ┌────▼──────┐  ┌───▼────────────┐
    │ Show login│  │ Check: Is user │
    │ prompt    │  │ a customer?    │
    │           │  │                │
    │ • Login   │  │ $_SESSION[     │
    │ • Register│  │ 'user_role']   │
    └───────────┘  │ == 'customer'? │
                   └───┬──────────┬──┘
                       │ NO       │ YES
                       │          │
                   ┌───▼──────┐ ┌─▼───────────┐
                   │ Show     │ │ Show all    │
                   │ admin    │ │ products &  │
                   │ message  │ │ allow       │
                   │ "Cannot  │ │ shopping    │
                   │ purchase"│ │             │
                   │          │ │ • Add to    │
                   │ • Logout │ │   cart ok   │
                   │          │ │ • Remove ok │
                   │ • Login  │ │             │
                   │   again  │ │ • All AJAX  │
                   └──────────┘ │   works     │
                                └─────────────┘
```

---

## 📊 Database Schema Relationships

```
┌─────────────────────┐         ┌──────────────────────────┐
│    users TABLE      │         │      cart TABLE          │
├─────────────────────┤         ├──────────────────────────┤
│ id (PK)         [1] │◄────┐   │ id (PK)                  │
│ name                │     └───│ user_id (FK) ────────┐  │
│ email               │         │ product_id           │  │
│ phone               │         │ product_name         │  │
│ password            │         │ price                │  │
│ role (admin/cust)   │         │ image                │  │
│ status              │         │ quantity             │  │
│ created_at          │         │ created_at           │  │
│ updated_at          │         │ updated_at           │  │
└─────────────────────┘         └──────────────────────────┘
        ▲
        │ ONE user can have
        │ MANY cart items
        │
        └─ Relationship: users.id = cart.user_id
           Constraint: ON DELETE CASCADE
           (deleting user deletes their cart)
```

---

## 🔄 Session & State Management

```
┌─────────────────────────────────────────────────────────────┐
│              Browser Session State                          │
└─────────────────────────────────────────────────────────────┘

After User Logs In:
┌─────────────────────────┐
│ $_SESSION = {           │
│   user_id: 5,           │◄─ Used for verification
│   user_role: "customer",│◄─ Used for authorization
│   user_name: "John",    │
│   login_time: ...       │
│ }                       │
└─────────────────────────┘
         │
         ▼
On Every AJAX Request:
┌─────────────────────────────────────────┐
│ cart-api.php checks:                    │
│                                         │
│ 1. if (!isset($_SESSION['user_id']))    │
│    ✗ Return 401 Unauthorized            │
│    ✓ Continue                           │
│                                         │
│ 2. if ($_SESSION['user_role'] !=        │
│    'customer')                          │
│    ✗ Return 403 Forbidden               │
│    ✓ Continue                           │
│                                         │
│ 3. If needed: Verify cart item          │
│    belongs to user                      │
│    SELECT... WHERE id=? AND             │
│    user_id=?                            │
│    ✗ Return 403 Unauthorized            │
│    ✓ Continue                           │
│                                         │
│ 4. Execute database query               │
│ 5. Return success response              │
└─────────────────────────────────────────┘
```

---

## 🛡️ Security Layers

```
REQUEST CHAIN:
Browser AJAX Request
    │
    ├─ Layer 1: HTTPS/TLS (in production)
    │           └─ Encrypts data in transit
    │
    ├─ Layer 2: Session Validation
    │           └─ Checks if user is logged in
    │
    ├─ Layer 3: Role Verification
    │           └─ Checks if user is customer
    │
    ├─ Layer 4: Input Validation
    │           └─ Checks data types & values
    │
    ├─ Layer 5: SQL Prepared Statements
    │           └─ Prevents SQL Injection
    │           └─ Binds parameters safely
    │
    ├─ Layer 6: Ownership Verification
    │           └─ Checks if item belongs to user
    │
    └─ Layer 7: Error Handling
                └─ Returns safe error messages
                └─ Logs sensitive data internally

Result: Secure, validated, database-safe request
```

---

## 📈 Request/Response Cycle

```
BROWSER              NETWORK              PHP SERVER
   │                    │                      │
   │─── POST request ───>│                     │
   │  action=add         │─ Parse request ──> PHP
   │  product_id=1       │                    │
   │  quantity=2         │                    ├─ Validate auth
   │                     │                    ├─ Check role
   │                     │                    ├─ Query database
   │                     │                    ├─ Execute insert
   │                     │                    ├─ Get cart count
   │                     │
   │<── JSON response ───│<── Build JSON ────│
   │  {success: true}    │                    │
   │  {cart_count: 5}    │                    │
   │                     │                    │
   └─ JavaScript ────────────────────────────┘
      ├─ Parse JSON
      ├─ Show toast
      ├─ Update UI
      └─ NO PAGE RELOAD
```

---

## 🧠 Data Flow Summary

```
User Action:
   │
   ├─ Click ADD TO CART
   │  └─ JavaScript event listener fires
   │     └─ Collects product data
   │        └─ Creates FormData object
   │           └─ Sends AJAX POST request
   │              └─ cart-api.php receives request
   │                 └─ PHP validates session
   │                    └─ PHP checks role
   │                       └─ PHP queries database
   │                          └─ PHP executes INSERT/UPDATE
   │                             └─ PHP gets new cart count
   │                                └─ PHP returns JSON
   │                                   └─ JavaScript receives response
   │                                      └─ Shows toast
   │                                         └─ Updates cart count
   │                                            └─ (NO RELOAD)

Total Time: ~200-500ms (invisible to user)
User Experience: Instant feedback ✅
```

---

## 🎯 Key Takeaways

1. **Real-Time**: AJAX prevents page reloads
2. **Secure**: Multiple validation layers
3. **Fast**: Minimal data transfer
4. **Responsive**: Touch-friendly UI
5. **Reliable**: Error handling throughout
6. **Scalable**: Architecture supports future features

---

## 📚 For Developers

### To Debug:
1. Open Browser DevTools (F12)
2. Go to Network tab
3. Click "Add to Cart"
4. Watch the POST request to `cart-api.php`
5. See the JSON response with cart count

### To Monitor:
1. Open Console tab (F12)
2. Watch for console.log() messages
3. Check for JavaScript errors
4. Monitor network requests

### To Modify:
1. All AJAX code is in `<script>` tags
2. API handling is in `cart-api.php`
3. Database schema in `migrate-cart.php`
4. Styling in `<style>` tags

---

That's the complete system architecture! 🎉
