# Blog System - Quick Reference

## 🚀 3-Step Quick Start

### 1️⃣ Setup Database (1 minute)
```
Visit: http://localhost/interior/setup-blogs-db.php
```

### 2️⃣ View Blog Page (1 minute)
```
Visit: http://localhost/interior/blog.php
```

### 3️⃣ Add Blog Posts (2 minutes)
```
Visit: http://localhost/interior/admin-dashboard.php → Blogs
```

---

## 📁 Key Files

| File | Purpose | What to Do |
|------|---------|-----------|
| `blog.php` | Public blog page | Just use it! Displays all blogs from DB |
| `api/blog-api.php` | API endpoints | Used by admin panel & frontend |
| `setup-blogs-db.php` | Database setup | Run once to create tables |
| `config/db.php` | DB connection | No changes needed |
| `admin-dashboard.php` | Admin panel | Already integrated! |

---

## 🔗 Important URLs

| URL | Purpose |
|-----|---------|
| `http://localhost/interior/setup-blogs-db.php` | Initial setup |
| `http://localhost/interior/blog.php` | View all blogs |
| `http://localhost/interior/admin-dashboard.php` | Add/edit blogs |
| `http://localhost/interior/api/blog-api.php?action=get-blogs` | Get all blogs (JSON) |

---

## 💾 Database Info

**Table**: `blogs`

**Columns**:
```
id              → Blog ID (auto-increment)
title           → Blog title
category        → Blog category
author          → Author name
day             → Publication day (01-31)
month           → Publication month (JANUARY, etc.)
image_path      → Image file path
content         → Full blog content (optional)
created_at      → Auto timestamp
updated_at      → Auto timestamp
```

---

## ✏️ Add Blog Post

### Via Admin Dashboard:
1. Go to admin-dashboard.php
2. Click "Blogs" section
3. Click "Add New Blog"
4. Fill form + upload image
5. Submit

### Via API:
```bash
curl -X POST http://localhost/interior/api/blog-api.php?action=add-blog \
  -F "title=My Blog" \
  -F "category=Interior" \
  -F "author=John Doe" \
  -F "day=25" \
  -F "month=JANUARY" \
  -F "image=@image.jpg"
```

### Via Database:
```sql
INSERT INTO blogs (title, category, author, day, month, image_path) 
VALUES ('Title', 'Category', 'Author', '25', 'JANUARY', 'path/to/image.jpg');
```

---

## 🔍 Search & Filter

### Search on Blog Page:
```html
<form action="blog.php" method="GET">
  <input name="search" placeholder="Search blogs...">
</form>
```

### API Search:
```
GET /api/blog-api.php?action=search-blogs&q=interior
```

### Filter by Category:
```
http://localhost/interior/blog.php?category=Interior
```

---

## 📊 API Response Examples

### Get All Blogs:
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "title": "Blog Title",
      "category": "Interior",
      "author": "John Doe",
      "day": "18",
      "month": "JUNE",
      "image_path": "assets/img/blog/...",
      "created_at": "2025-01-01 10:00:00"
    }
  ],
  "count": 3
}
```

### Get Single Blog:
```json
{
  "success": true,
  "data": {
    "id": 1,
    "title": "Blog Title",
    "category": "Interior",
    "author": "John Doe",
    "day": "18",
    "month": "JUNE",
    "image_path": "assets/img/blog/...",
    "content": "Full blog content here...",
    "created_at": "2025-01-01 10:00:00"
  }
}
```

---

## 🛠️ Common Tasks

### Display Blogs in PHP:
```php
<?php
require 'config/db.php';
$query = "SELECT * FROM blogs ORDER BY created_at DESC";
$result = $connection->query($query);
while ($blog = $result->fetch_assoc()) {
    echo "<h2>" . htmlspecialchars($blog['title']) . "</h2>";
}
?>
```

### Get Blogs via JavaScript:
```javascript
fetch('/interior/api/blog-api.php?action=get-blogs')
  .then(r => r.json())
  .then(data => {
    data.data.forEach(blog => {
      console.log(blog.title);
    });
  });
```

### Update Blog:
```php
// Via form to api/blog-api.php?action=update-blog
$stmt = $connection->prepare("UPDATE blogs SET title=?, category=? WHERE id=?");
$stmt->bind_param('ssi', $title, $category, $id);
$stmt->execute();
```

### Delete Blog:
```php
// Via form to api/blog-api.php?action=delete-blog&id=1
$stmt = $connection->prepare("DELETE FROM blogs WHERE id=?");
$stmt->bind_param('i', $id);
$stmt->execute();
```

---

## 🐛 Troubleshooting

### Problem: "Table doesn't exist"
```
Solution: Run setup-blogs-db.php
```

### Problem: Images not uploading
```
Solution: Check /assets/img/blog/ is writable
mkdir -p assets/img/blog
chmod 755 assets/img/blog
```

### Problem: Empty blog page
```
Solution: 
1. Run setup-blogs-db.php
2. Add some blog posts
3. Refresh blog.php
```

### Problem: Database connection error
```
Solution: Check config/db.php settings
DB_HOST = 'localhost'
DB_USER = 'root'
DB_PASSWORD = ''
DB_NAME = 'interior_design'
```

---

## 🔐 Security Notes

✅ All inputs validated
✅ Prepared statements used
✅ File type checking (images only)
✅ File size limit (5MB max)
✅ SQL injection prevention
✅ XSS protection with htmlspecialchars()
✅ CSRF tokens (if using forms)

---

## 📱 Mobile Responsive

Blog page is fully responsive:
- ✅ Desktop (1024px+)
- ✅ Tablet (768-1024px)
- ✅ Mobile (480-768px)
- ✅ Small phones (360-480px)

---

## 🎨 Customization

### Change blog card style:
Edit CSS in `blog.php` lines 400-500

### Change blog layout:
Edit HTML grid in `blog.php` lines 600-650

### Add new fields:
```sql
ALTER TABLE blogs ADD COLUMN new_field VARCHAR(255);
```

### Change items per page:
In `blog.php`, use `LIMIT 10 OFFSET 0`

---

## 📈 Performance

- Indexed on: category, author, created_at
- Pagination ready (add LIMIT clause)
- Query caching ready (implement with memcached)
- Image optimization recommended (compress before upload)

---

## 🚨 Limits & Rules

| Item | Limit |
|------|-------|
| Title length | 255 characters |
| Category length | 100 characters |
| Author length | 100 characters |
| Image size | 5 MB max |
| Image types | JPEG, PNG, GIF, WebP only |
| Content length | Unlimited |

---

## 🔄 Update Cycle

1. **Add Blog** → Saved to database
2. **Blog Page** → Fetches from DB
3. **Admin Panel** → Shows all blogs
4. **API** → Returns JSON data
5. **Frontend** → Displays to users

---

## 💡 Best Practices

1. Always use `htmlspecialchars()` when displaying data
2. Use prepared statements for all queries
3. Validate file uploads
4. Optimize images before uploading
5. Use appropriate category names
6. Include author information
7. Add meaningful content
8. Test on mobile devices
9. Regular database backups
10. Monitor performance

---

## 📞 Need Help?

1. Check BLOG_DATABASE_INTEGRATION.md (detailed guide)
2. Review admin-dashboard.php (existing implementation)
3. Check phpMyAdmin for database issues
4. Test API endpoints with browser
5. Review error logs

---

## ✅ Implementation Checklist

- [x] Database table created
- [x] blog.php created
- [x] API endpoints created
- [x] Admin integration done
- [x] Sample data added
- [x] Security implemented
- [x] Documentation created
- [x] Responsive design done
- [x] Error handling added
- [x] Testing ready

---

**Status**: ✅ Ready to Use!

All blog functionality is integrated with the database. Start adding blog posts now!
