# Customer Dashboard & Consultation System - Quick Reference

## 🎯 What's Included

### 1. Customer Dashboard (`customer-dashboard.php`)
Modern dashboard with 4 tabs:
- **Overview** - Welcome & statistics
- **Orders** - Real-time order display from database
- **Profile** - Edit name, phone, address, change password
- **Schedule Consultation** - Book consultation with form

### 2. Consultation System
Complete end-to-end consultation booking:
- **Form** - Multi-section form with validation
- **Email Verification** - Token-based email confirmation
- **Admin Notifications** - Admin gets notified of new requests
- **Status Tracking** - pending → verified → scheduled

### 3. API Endpoints
- `api/update-profile.php` - Update profile & password
- `api/submit-consultation.php` - Submit & email verification

## ⚡ Quick Start (5 minutes)

### 1. Create Database Table
Visit in your browser:
```
http://localhost/interior/setup-consultations-db.php
```

Or manually:
```sql
-- Copy and paste into phpMyAdmin
CREATE TABLE consultations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    email VARCHAR(255),
    phone VARCHAR(20),
    address TEXT,
    consultation_type VARCHAR(50),
    preferred_date DATE,
    preferred_time TIME,
    project_description TEXT,
    budget_range VARCHAR(50),
    source VARCHAR(50),
    verification_token VARCHAR(255),
    token_expiry DATETIME,
    status ENUM('pending', 'verified', 'scheduled', 'completed', 'cancelled'),
    verified_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);
```

### 2. Update Email Config
Edit `config/email.php`:
```php
define('MAIL_FROM_EMAIL', 'your-email@yoursite.com');
define('ADMIN_EMAIL', 'admin@yoursite.com');
```

### 3. Access Pages
- Dashboard: `http://localhost/interior/customer-dashboard.php`
- Consultation: `http://localhost/interior/consultation.php`

## 📊 Features Overview

### Dashboard Statistics
Automatic calculations from database:
- Total Orders
- Completed Orders
- Pending Orders
- Total Spent (sum of order amounts)

### Order Display
Real-time order table showing:
- Order ID
- Order Date
- Amount
- Status (color-coded badges)
- View Details link

### Profile Management
Update form with:
- Full Name
- Phone Number
- Address
- Password Change (with verification)

### Consultation Form
Comprehensive form with:
- Personal Information (name, email, phone, address)
- Consultation Type (5 options)
- Date & Time Selection
- Project Description
- Budget Range (5 options)
- How you heard about us

## 🔄 Consultation Flow

```
1. User fills form
   ↓
2. Server validates all fields
   ↓
3. Generate 32-byte verification token
   ↓
4. Save to database with 24-hour expiry
   ↓
5. Send verification email with link
   ↓
6. Send admin notification
   ↓
7. Show success modal
   ↓
8. User clicks email link
   ↓
9. Verify token & update status to "verified"
   ↓
10. Show success page with consultation details
```

## 🔐 Security

- ✅ Session-based authentication
- ✅ Password hashing with bcrypt
- ✅ SQL injection prevention
- ✅ Email verification tokens
- ✅ 24-hour token expiry
- ✅ Server-side validation
- ✅ Prepared statements for all queries

## 📱 Responsive Design

All pages are fully responsive:
- Desktop (1200px+)
- Tablet (768px - 1199px)
- Mobile (< 768px)

## 🎨 Styling

Beautiful gradient backgrounds:
- Primary: Purple gradient (#667eea → #764ba2)
- Status badges: Color-coded
- Form validation: Real-time feedback
- Smooth animations and transitions

## 🧪 Test Checklist

### Before Going Live
1. [ ] Create consultations table
2. [ ] Configure email settings
3. [ ] Test profile update
4. [ ] Test consultation submission
5. [ ] Check verification email
6. [ ] Verify email link works
7. [ ] Confirm status updated
8. [ ] Check admin email received
9. [ ] Test on mobile device
10. [ ] Check database for records

## 📋 Database Queries

### View All Consultations
```sql
SELECT * FROM consultations ORDER BY created_at DESC;
```

### View Pending Consultations
```sql
SELECT * FROM consultations WHERE status = 'pending';
```

### View Verified Consultations
```sql
SELECT * FROM consultations WHERE status = 'verified';
```

### Update Consultation Status
```sql
UPDATE consultations SET status = 'scheduled', scheduled_at = NOW() WHERE id = 1;
```

## 🛠️ File Structure

```
interior/
├── customer-dashboard.php          (Main dashboard)
├── consultation.php                (Consultation form)
├── verify-consultation.php         (Email verification)
├── api/
│   ├── update-profile.php         (Profile API)
│   └── submit-consultation.php    (Consultation API)
├── config/
│   ├── db.php                     (Database connection)
│   └── email.php                  (Email config)
├── setup-consultations-db.php     (Database setup)
├── auth-system/database/
│   └── create_consultations_table.sql
└── CUSTOMER_DASHBOARD_SETUP.md    (Full guide)
```

## ⚠️ Common Issues

### Email Not Sending
1. Check `config/email.php` email address
2. Verify SMTP settings in `php.ini`
3. Check server error logs
4. Test with `mail()` function directly

### Verification Link Not Working
1. Ensure database table exists
2. Check token in verification URL
3. Verify token hasn't expired
4. Test with fresh token

### Profile Update Not Working
1. Verify session is active
2. Check password verification
3. Test database connectivity
4. Check form field names match

## 📞 Need Help?

Check these files for detailed info:
- **Full Setup Guide:** `CUSTOMER_DASHBOARD_SETUP.md`
- **Database Issues:** `config/db.php` 
- **Email Issues:** `config/email.php`
- **API Issues:** `api/*.php` files

## 🚀 Next Steps

1. ✅ Database table created
2. ✅ Pages built and tested
3. ⏭️ Admin consultation panel (optional)
4. ⏭️ Booking calendar system (optional)
5. ⏭️ SMS notifications (optional)

---

**Quick Test URL:**
- Dashboard: `http://localhost/interior/customer-dashboard.php`
- Consultation: `http://localhost/interior/consultation.php`
- DB Setup: `http://localhost/interior/setup-consultations-db.php`

**All pages require login!** Make sure user is logged in before accessing.
