# Customer Dashboard & Consultation System - Complete Setup Guide

## Overview
This guide covers the complete setup of the Customer Dashboard and Consultation System, including the database schema, API endpoints, and verification system.

## 📋 Files Created/Modified

### 1. **Database Setup**
- `setup-consultations-db.php` - Automated database setup script
- `auth-system/database/create_consultations_table.sql` - SQL migration file

### 2. **Configuration**
- `config/email.php` - Email configuration and helper functions

### 3. **API Endpoints**
- `api/update-profile.php` - User profile update with password change
- `api/submit-consultation.php` - Consultation submission with email verification

### 4. **User-Facing Pages**
- `customer-dashboard.php` - Main dashboard with orders, profile, and consultation management
- `consultation.php` - Standalone consultation booking page
- `verify-consultation.php` - Email verification landing page

## 🗄️ Database Schema

### Consultations Table
```sql
CREATE TABLE consultations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    address TEXT NOT NULL,
    consultation_type VARCHAR(50) NOT NULL,
    preferred_date DATE NOT NULL,
    preferred_time TIME NOT NULL,
    project_description TEXT NOT NULL,
    budget_range VARCHAR(50) NOT NULL,
    source VARCHAR(50),
    verification_token VARCHAR(255),
    token_expiry DATETIME,
    status ENUM('pending', 'verified', 'scheduled', 'completed', 'cancelled') DEFAULT 'pending',
    verified_at DATETIME,
    scheduled_at DATETIME,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_email (email),
    INDEX idx_verification_token (verification_token)
);
```

## 🚀 Setup Steps

### Step 1: Create Database Table
Visit `http://localhost/interior/setup-consultations-db.php` in your browser to automatically create the consultations table.

**Or manually execute the SQL:**
```bash
mysql -u root -p interior < auth-system/database/create_consultations_table.sql
```

### Step 2: Configure Email Settings
Edit `config/email.php` and update:

```php
define('MAIL_FROM_NAME', 'Interior Design Solutions');
define('MAIL_FROM_EMAIL', 'noreply@yourinteriorsite.com');
define('ADMIN_EMAIL', 'admin@yourinteriorsite.com');

// For SMTP (optional):
// define('SMTP_HOST', 'smtp.gmail.com');
// define('SMTP_PORT', 587);
// define('SMTP_USERNAME', 'your-email@gmail.com');
// define('SMTP_PASSWORD', 'your-app-password');
```

### Step 3: Update Users Table (if needed)
Ensure the `users` table has `phone` and `address` columns:

```sql
ALTER TABLE users ADD COLUMN phone VARCHAR(20) AFTER email;
ALTER TABLE users ADD COLUMN address TEXT AFTER phone;
```

## 📱 Features

### Customer Dashboard (`customer-dashboard.php`)

**Tabs:**
1. **Overview** - Welcome section with quick stats
2. **Orders** - Display all customer orders with status and details
3. **Profile** - Edit personal information and change password
4. **Schedule Consultation** - Book a consultation with form validation

**Statistics Displayed:**
- Total Orders
- Completed Orders
- Pending Orders
- Total Amount Spent

**Profile Management:**
- Update name, phone, address
- Change password with current password verification
- Async form submission with validation

### Consultation System

**Step 1: Submission**
- User fills consultation form with personal and project details
- Form validates on client and server side
- System generates 32-byte hex token

**Step 2: Email Verification**
- Verification email sent with 24-hour expiry link
- Email contains consultation details
- Success modal shown to user

**Step 3: Verification**
- User clicks email verification link
- Token validated against database
- Status updated to 'verified'
- Success page displays consultation details

**Step 4: Admin Notification**
- Admin receives email notification
- Can view all consultations in admin panel

## 🔐 Security Features

1. **Session-Based Authentication**
   - All pages check `$_SESSION['user_id']`
   - Redirect to login if not authenticated

2. **Password Security**
   - Passwords hashed with `PASSWORD_BCRYPT`
   - Verified with `password_verify()`
   - Current password required to change

3. **Token Verification**
   - 32-byte random hex token: `bin2hex(random_bytes(32))`
   - 24-hour expiry time
   - One-time use (status updated on verification)

4. **Input Validation**
   - Server-side validation for all inputs
   - Email validation with `filter_var()`
   - SQL injection prevention with prepared statements

5. **CSRF Protection**
   - Consider adding CSRF tokens for production

## 📊 API Endpoints

### 1. Update Profile
**Endpoint:** `POST /api/update-profile.php`

**Parameters:**
```json
{
  "name": "John Doe",
  "phone": "+63912345678",
  "address": "123 Main St, City",
  "current_password": "old_password",
  "new_password": "new_password",
  "confirm_password": "new_password"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Profile updated successfully"
}
```

### 2. Submit Consultation
**Endpoint:** `POST /api/submit-consultation.php`

**Parameters:**
```json
{
  "first_name": "John",
  "last_name": "Doe",
  "email": "john@example.com",
  "phone": "+63912345678",
  "address": "123 Main St",
  "consultation_type": "residential",
  "preferred_date": "2024-01-15",
  "preferred_time": "14:00",
  "project_description": "I want to redesign my living room...",
  "budget_range": "100k-200k",
  "source": "google"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Consultation request submitted successfully!",
  "consultation_id": 123
}
```

## 🔧 Configuration Details

### Email Configuration (`config/email.php`)

**Default Settings:**
- FROM: Interior Design Solutions <noreply@yourinteriorsite.com>
- ADMIN: admin@yourinteriorsite.com
- METHOD: PHP mail()

**For Production (SMTP):**
Use PHPMailer or configure SMTP with:
```php
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password');
define('SMTP_ENCRYPTION', 'tls');
```

## 📧 Email Templates

### Consultation Verification Email
- Sent to customer with verification link
- Link expires in 24 hours
- HTML formatted with company branding
- Includes verification instructions

### Admin Notification Email
- Sent to admin on consultation submission
- Contains all consultation details
- Shows pending verification status
- Used for follow-up management

## 🧪 Testing

### Test Consultation Flow
1. Navigate to `customer-dashboard.php`
2. Click "Schedule Consultation" tab
3. Fill form with test data
4. Submit form
5. Check for verification email
6. Click verification link
7. Verify status updated in database

### Test Profile Update
1. Navigate to `customer-dashboard.php`
2. Click "Profile" tab
3. Update name, phone, address
4. Save changes
5. Verify changes in database

## 🐛 Troubleshooting

### Email Not Sending
1. Check `php.ini` SMTP settings
2. Verify MAIL_FROM_EMAIL is correct
3. Check server mail logs
4. Enable error reporting in `config/email.php`

### Verification Link Not Working
1. Check token in database matches URL
2. Verify token_expiry hasn't passed
3. Ensure consultations table exists
4. Check for SQL errors in logs

### 404 on API Endpoints
1. Ensure `/api` directory exists
2. Verify `.htaccess` doesn't block API
3. Check file permissions

## 📈 Future Enhancements

1. **Admin Panel**
   - View all consultations
   - Manage consultation status
   - Schedule confirmed appointments
   - Send follow-up emails

2. **Email Templates**
   - Customizable templates
   - Template variables for personalization
   - Multi-language support

3. **Notifications**
   - SMS notifications
   - Push notifications
   - In-app notifications

4. **Analytics**
   - Consultation conversion rates
   - Popular consultation types
   - Customer satisfaction surveys

5. **Integration**
   - Google Calendar sync
   - Calendar API integration
   - Payment processing for deposits

## 🔗 Related Files

- [Customer Dashboard Code](./customer-dashboard.php)
- [Consultation Booking Code](./consultation.php)
- [API Update Profile](./api/update-profile.php)
- [API Submit Consultation](./api/submit-consultation.php)
- [Email Configuration](./config/email.php)
- [Verification Page](./verify-consultation.php)

## ✅ Verification Checklist

- [ ] Database table created
- [ ] Email configuration updated
- [ ] Users table has phone and address columns
- [ ] All pages accessible to logged-in users
- [ ] Profile update working
- [ ] Consultation form submitting
- [ ] Verification email sending
- [ ] Email verification link working
- [ ] Status updating to verified
- [ ] Admin receiving notifications

## 📞 Support

For issues or questions:
1. Check error logs in browser console
2. Review PHP error logs
3. Test database queries directly
4. Verify all files are in correct locations

---

**Last Updated:** 2024
**Status:** Complete and Ready for Testing
