# 📧 GMAIL SMTP SETUP - Email Not Receiving Fix

## Problem
Emails not being sent to customers when approving/declining consultations.

## Root Cause
Windows XAMPP's `mail()` function is unreliable. We need proper SMTP.

---

## ✅ SOLUTION: Use Gmail SMTP (Free & Reliable)

### Step 1: Get Your Gmail App Password

1. **Go to Google Account Security:**
   https://myaccount.google.com/security

2. **Enable 2-Step Verification:**
   - Click "2-Step Verification"
   - Follow steps to enable it (you'll need your phone)

3. **Generate App Password:**
   - Go back to https://myaccount.google.com/security
   - Scroll down to "App passwords" section
   - Select "Mail" and "Windows Computer"
   - Google gives you a 16-character password
   - **Copy this password** (you'll use it in Step 3)

### Step 2: Download PHPMailer

Run this batch file (creates vendor folder automatically):

**Create file: `C:\xampp\htdocs\interior\install-phpmailer.bat`**

```batch
@echo off
cd /d C:\xampp\htdocs\interior
echo Installing PHPMailer via Composer...
composer require phpmailer/phpmailer
pause
```

Then run it:
1. Right-click `install-phpmailer.bat`
2. Click "Run as Administrator"
3. Wait for installation complete

**OR** if Composer not installed:

```batch
@echo off
cd /d C:\xampp\htdocs\interior
mkdir vendor\phpmailer\phpmailer\src

REM Download PHPMailer manually
echo Please download from: https://github.com/PHPMailer/PHPMailer/archive/v6.9.1.zip
echo Extract to: C:\xampp\htdocs\interior\vendor\phpmailer\phpmailer
pause
```

### Step 3: Create Configuration File

**Create: `C:\xampp\htdocs\interior\config\mail.php`**

```php
<?php
// Gmail SMTP Configuration
return [
    'SMTP_HOST' => 'smtp.gmail.com',
    'SMTP_PORT' => 587,
    'SMTP_USER' => 'your-gmail@gmail.com',  // ← YOUR GMAIL ADDRESS
    'SMTP_PASS' => 'xxxx xxxx xxxx xxxx',   // ← THE 16-CHAR PASSWORD FROM STEP 1
    'FROM_EMAIL' => 'noreply@interiordesign.com',
    'FROM_NAME' => 'Interior Design Solutions'
];
?>
```

**IMPORTANT:** 
- Replace `your-gmail@gmail.com` with your actual Gmail
- Replace `xxxx xxxx xxxx xxxx` with the 16-char password from Step 1
- Keep spaces in the app password (it's: word space word space word space word)

### Step 4: Load Configuration in EmailService

The `EmailService.php` class already handles this. Just make sure the config file exists.

### Step 5: Update Approve/Decline APIs

Update both files to use EmailService instead of mail():

**File: `api/approve-consultation.php`**
- Replace `@mail()` call with EmailService
- Already implemented in updated version

**File: `api/decline-consultation.php`**
- Replace `@mail()` call with EmailService  
- Already implemented in updated version

---

## 🧪 Test It Works

### Quick Test:

1. Create this test file: `C:\xampp\htdocs\interior\test-email-setup.php`

```php
<?php
require_once 'classes/EmailService.php';

echo "<h2>Testing Email Setup</h2>";

$emailService = new EmailService();

// Test SMTP connection
$test = $emailService->testConnection();

if ($test['success']) {
    echo "<p style='color: green;'><strong>✓ SMTP Connection: SUCCESS</strong></p>";
    
    // Test sending approval email
    $result = $emailService->sendApprovalEmail(
        'your-test-email@gmail.com',  // ← CHANGE THIS
        'John Doe',
        'kitchen_renovation',
        '2026-02-15',
        '$5,000 - $10,000'
    );
    
    if ($result['success']) {
        echo "<p style='color: green;'><strong>✓ Approval Email: SENT</strong></p>";
        echo "<p>Check your email inbox (including spam folder)</p>";
    } else {
        echo "<p style='color: red;'><strong>✗ Error: " . $result['message'] . "</strong></p>";
    }
} else {
    echo "<p style='color: red;'><strong>✗ SMTP Connection Failed</strong></p>";
    echo "<p>Error: " . $test['message'] . "</p>";
    echo "<p><strong>Troubleshooting:</strong></p>";
    echo "<ul>";
    echo "<li>Verify Gmail address is correct</li>";
    echo "<li>Verify 16-char app password is correct (with spaces)</li>";
    echo "<li>Check 2-Step Verification is enabled</li>";
    echo "<li>Verify PHPMailer is installed in vendor folder</li>";
    echo "</ul>";
}
?>
```

2. Visit: `http://localhost/interior/test-email-setup.php`
3. Check inbox for test email

---

## 📋 Checklist

- [ ] 2-Step Verification enabled on Gmail
- [ ] App password generated (16 characters)
- [ ] PHPMailer installed in `vendor/` folder
- [ ] `config/mail.php` created with correct credentials
- [ ] `classes/EmailService.php` exists
- [ ] Test email sent successfully
- [ ] Real approval/decline tested
- [ ] Customer received email ✅

---

## 🚨 Common Issues & Fixes

### **Issue: "SMTP Error: Could not authenticate"**
**Solution:**
- Check that 2-Step Verification IS enabled
- Verify you're using App Password (16 chars), NOT regular Gmail password
- Make sure there are spaces in the app password

### **Issue: "SMTP Error: Could not connect to host"**
**Solution:**
- Verify `smtp.gmail.com` is in config
- Check SMTP port is 587
- Make sure PHPMailer is installed
- Restart Apache

### **Issue: "Class EmailService not found"**
**Solution:**
- Verify file exists: `C:\xampp\htdocs\interior\classes\EmailService.php`
- Check `vendor/autoload.php` exists
- Verify PHPMailer properly installed

### **Issue: Email still not arriving**
**Solution:**
- Check spam folder
- Verify recipient email is correct
- Check error logs in `error.log`
- Run test-email-setup.php to diagnose

---

## 🔄 Implementation Steps

1. **Follow Steps 1-4 above** (Gmail setup)
2. **Verify files exist:**
   - ✓ `classes/EmailService.php`
   - ✓ `config/mail.php`
   - ✓ `vendor/autoload.php`

3. **Run test:**
   - Visit test-email-setup.php
   - Should see green "SUCCESS" messages
   - Check email inbox

4. **Test in production:**
   - Go to admin dashboard
   - Try approving/declining a consultation
   - Customer should receive email within seconds

---

## 📞 Alternative: Use SendGrid (if Gmail doesn't work)

SendGrid is FREE for up to 100 emails/day:

1. **Sign up:** https://sendgrid.com (free account)
2. **Get API Key** from settings
3. **Update `config/mail.php`:**

```php
<?php
return [
    'SMTP_HOST' => 'smtp.sendgrid.net',
    'SMTP_PORT' => 587,
    'SMTP_USER' => 'apikey',  // Always "apikey"
    'SMTP_PASS' => 'SG.xxxxxxxxxxxxx',  // Your SendGrid API key
    'FROM_EMAIL' => 'noreply@interiordesign.com',
    'FROM_NAME' => 'Interior Design Solutions'
];
?>
```

---

## ✅ Once Working

After setup is complete:

✅ Approval emails send automatically
✅ Decline emails send automatically
✅ Emails delivered within seconds
✅ Professional HTML templates
✅ Customer gets instant confirmation

---

## 📊 What's Changed

**Before:**
- ❌ Using unreliable `mail()` function
- ❌ Emails not sent on Windows XAMPP
- ❌ Customer confused about status

**After:**
- ✅ Using Gmail SMTP (reliable)
- ✅ Emails sent reliably every time
- ✅ Customer gets instant confirmation
- ✅ Professional email with details

---

## 🎯 Quick Setup (5 minutes)

1. Get 16-char app password from Gmail (2 min)
2. Create `config/mail.php` (1 min)
3. Install PHPMailer if needed (1 min)
4. Test with test-email-setup.php (1 min)

**Total time: ~5 minutes**

---

**Need Help?** Check the Common Issues section above.

**Status:** ✅ Email system will be FIXED and WORKING!
