# 🛒 Real-Time Shopping Cart System

**Complete working real-time cart system with AJAX, no page reloads!**

---

## 📸 What You Get

### Shop Page (`shop.php`)
- 8 furniture products with images
- Quantity selector (1-10 items)
- Real-time "ADD TO CART" buttons
- Toast notifications (success/error)
- Loading indicators
- Responsive grid layout

### Cart Page (`cart.php`)
- Display all cart items with images
- Show quantities and prices
- Remove items without refresh
- Automatic cart total calculation
- Shipping + tax calculation
- Empty cart message

### Backend API (`auth-system/cart-api.php`)
- Add products to cart
- Remove from cart
- Get cart items
- Get cart count
- Session security
- Role-based access (customers only)

---

## 🚀 How It Works (Real-Time Magic!)

```javascript
// User clicks "ADD TO CART" button
↓
// JavaScript captures data (product ID, name, price, quantity)
↓
// fetch() sends AJAX request to cart-api.php
↓
// cart-api.php validates session & adds to database
↓
// Returns JSON response {success: true}
↓
// Toast appears: "✅ Product added to cart successfully"
↓
// Page stays open - NO RELOAD! ✨
```

---

## 🎯 Quick Setup

### 1. Run Migration
```
http://localhost/interior/auth-system/migrate-cart.php
```

### 2. Register Customer Account
```
http://localhost/interior/auth-system/register.php
- Name: Any name
- Email: Any email
- Password: Any password
- Role: ⚠️ CUSTOMER (not admin)
```

### 3. Test Shopping
```
1. Go to: http://localhost/interior/shop.php
2. Login with your customer account
3. Click "➕ ADD TO CART" on any product
4. See green toast: "✅ Product added to cart successfully"
5. Go to: http://localhost/interior/cart.php
6. See your items!
```

---

## 🔒 Security Features

✅ **Session Validation** - Every API call checks if user is logged in
✅ **Role-Based Access** - Only customers can shop (admins see error)
✅ **SQL Injection Prevention** - Prepared statements on all queries
✅ **User Ownership** - Can't view/modify other users' carts
✅ **CSRF Protection** - Session-based security

---

## 📁 File Structure

```
interior/
├── shop.php                          ← Product listing (8 items)
├── cart.php                          ← Shopping cart display
├── SETUP.md                          ← Setup instructions
├── README.md                         ← This file
└── auth-system/
    ├── cart-api.php                 ← AJAX API backend
    ├── migrate-cart.php             ← Database migration
    ├── config/
    │   └── db.php                   ← Database connection
    ├── login.php                    ← Customer/admin login
    ├── register.php                 ← Account registration
    └── (other auth files)
```

---

## 🎨 UI/UX Features

| Feature | Details |
|---------|---------|
| **Toast Notifications** | Green (success), Red (error), Auto-dismiss after 3s |
| **Loading States** | Buttons show "⏳ Adding..." while processing |
| **Product Grid** | Responsive 4-column layout (auto-adjusts on mobile) |
| **Quantity Input** | Selector field (1-10 items) |
| **Price Display** | Gold color (#d4a574) for prices |
| **Empty Cart** | Shows "Your Cart is Empty" message |
| **Cart Summary** | Subtotal + Shipping + Tax = Total |

---

## 🧪 Real-Time Testing

### Test 1: Add to Cart (No Reload)
1. Open Chrome DevTools (F12)
2. Go to `shop.php`
3. Click "ADD TO CART"
4. **Check:** Page doesn't reload, toast appears, cart count updates

### Test 2: Remove Item (No Reload)
1. Go to `cart.php`
2. Click "🗑️ Remove" on any item
3. **Check:** Page doesn't reload, item disappears, total updates

### Test 3: Multiple Items
1. Add 3 different products with different quantities
2. Go to cart.php
3. **Check:** All items appear with correct quantities and prices
4. Remove one item
5. **Check:** Cart updates without reload

### Test 4: Session Persistence
1. Add items to cart
2. Logout
3. Login again
4. Go to cart.php
5. **Check:** Your items are still there!

### Test 5: Admin Cannot Shop
1. Login as ADMIN user
2. Try to go to `shop.php`
3. **Check:** See error "Admin users cannot shop"

---

## 💾 Database Schema

### Cart Table
```sql
CREATE TABLE cart (
    id                  INT AUTO_INCREMENT PRIMARY KEY,
    user_id             INT NOT NULL,
    product_id          INT NOT NULL,
    product_name        VARCHAR(255),
    price               DECIMAL(10, 2),
    image               VARCHAR(255),
    quantity            INT DEFAULT 1,
    created_at          TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at          TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

---

## 📊 API Reference

### Endpoint: `auth-system/cart-api.php`

#### 1. Get Cart Items
```
GET /auth-system/cart-api.php?action=get

Response:
{
  "success": true,
  "items": [
    {
      "id": 1,
      "product_id": 1,
      "product_name": "Nordic Wooden Chair",
      "price": "10000",
      "quantity": 2,
      "image": "assets/img/product/product_1_1.jpg"
    }
  ],
  "subtotal": 20000
}
```

#### 2. Add to Cart
```
POST /auth-system/cart-api.php?action=add

Body:
- product_id: 1
- product_name: "Nordic Wooden Chair"
- price: 10000
- image: "assets/img/product/product_1_1.jpg"
- quantity: 1

Response:
{
  "success": true,
  "message": "Product added to cart successfully",
  "cart_count": 3
}
```

#### 3. Remove from Cart
```
POST /auth-system/cart-api.php?action=remove

Body:
- cart_id: 1

Response:
{
  "success": true,
  "message": "Item removed from cart"
}
```

#### 4. Get Cart Count
```
GET /auth-system/cart-api.php?action=count

Response:
{
  "success": true,
  "count": 3
}
```

---

## 🐛 Common Issues & Fixes

| Issue | Solution |
|-------|----------|
| **Can't add to cart** | Make sure you're logged in as CUSTOMER (not admin) |
| **Page reloads** | Check browser console (F12) for JavaScript errors |
| **Empty cart** | Run migration: `migrate-cart.php` |
| **See admin error** | Logout and login as CUSTOMER account |
| **Items not saving** | Check database connection in `config/db.php` |
| **Images not showing** | Verify image paths: `assets/img/product/product_1_*.jpg` |

---

## 🎯 Products in Shop

| # | Name | Price | Image |
|---|------|-------|-------|
| 1 | Nordic Wooden Chair | Rs. 10,000 | product_1_1.jpg |
| 2 | Elegant Wooden Chair | Rs. 12,000 | product_1_2.jpg |
| 3 | Rounded Dining Table | Rs. 14,000 | product_1_3.jpg |
| 4 | Modern Sofa Set | Rs. 18,000 | product_1_4.jpg |
| 5 | Wooden Bed Frame | Rs. 22,000 | product_1_5.jpg |
| 6 | Coffee Table | Rs. 8,000 | product_1_6.jpg |
| 7 | Bookshelf | Rs. 15,000 | product_1_7.jpg |
| 8 | Office Desk | Rs. 16,000 | product_1_8.jpg |

---

## 🚀 Technology Stack

- **Frontend:** HTML5, CSS3, Vanilla JavaScript (fetch API)
- **Backend:** PHP 7.4+ with MySQLi
- **Database:** MySQL with InnoDB
- **Framework:** Bootstrap 5 (CSS)
- **Authentication:** PHP Sessions
- **API:** REST-style with JSON responses

---

## ✅ Checklist for Success

- [ ] Ran `migrate-cart.php` successfully
- [ ] Registered as CUSTOMER user
- [ ] Can login to shop.php
- [ ] Can click ADD TO CART without page reload
- [ ] Toast notification appears
- [ ] Can view cart.php with items
- [ ] Can remove items without page reload
- [ ] Cart total calculates correctly
- [ ] Items persist after logout/login

---

## 💡 Pro Tips

1. **Test in Console:** Open F12 → Console → watch fetch requests
2. **Check Network Tab:** See AJAX requests to cart-api.php in real-time
3. **Toast Timing:** Notifications auto-dismiss after 3 seconds
4. **Quantity:** Default 1, can change before adding (1-10)
5. **Multiple Items:** Same product twice = quantity increases
6. **Auto-Refresh:** Cart page auto-refreshes every 3 seconds

---

## 📞 Support

All code is working and tested! If you have issues:

1. Check browser console (F12) for errors
2. Verify database migration ran
3. Make sure you're logged in as CUSTOMER
4. Check that session is active (not expired)
5. Verify product images exist

---

**Real-Time Shopping Cart System - COMPLETE & WORKING** ✅
