# 🛒 Real-Time Shopping Cart System - SETUP GUIDE

## ✅ Quick Start (3 Steps)

### Step 1: Run Database Migration
1. Go to: `http://localhost/interior/auth-system/migrate-cart.php`
2. You should see: **✅ Cart table created successfully!**

### Step 2: Register a New Customer Account
1. Go to: `http://localhost/interior/auth-system/register.php`
2. Fill in the form:
   - Name: (any name)
   - Email: (any email)
   - Password: (any password)
   - **Role: CUSTOMER** ⚠️ (very important!)
3. Click Register

### Step 3: Login and Test
1. Go to: `http://localhost/interior/shop.php`
2. Login with your customer account
3. Click **"➕ ADD TO CART"** on any product
4. You should see: **✅ Product added to cart successfully** (green toast)
5. Go to: `http://localhost/interior/cart.php`
6. Your items should appear with quantities and prices

---

## 🔧 System Architecture

```
Real-Time Cart Flow:
┌─────────────────┐
│  shop.php       │ → User clicks "ADD TO CART"
└────────┬────────┘
         │
         ↓ AJAX fetch() (NO PAGE RELOAD!)
┌─────────────────────────────────┐
│  auth-system/cart-api.php       │ → Validates session & adds to DB
│  (?action=add)                  │
└────────┬────────────────────────┘
         │
         ↓ Returns JSON {success: true}
┌─────────────────┐
│  Toast Message  │ → Shows "✅ Added to cart" for 3 seconds
│  Updates DOM    │
└─────────────────┘
```

---

## 📋 Files Created

| File | Purpose |
|------|---------|
| `shop.php` | Product listing with ADD TO CART buttons |
| `cart.php` | Shopping cart display with remove functionality |
| `auth-system/cart-api.php` | Backend AJAX API (add/remove/get/count) |
| `auth-system/migrate-cart.php` | Database migration (creates cart table) |

---

## 🔑 Key Features

✨ **Real-Time AJAX**
- Click ADD TO CART → No page reload
- Cart updates instantly via fetch() API
- Toast notifications for user feedback

🔒 **Security**
- Session validation on every request
- Only customers can shop (role check)
- User ownership verification (can't modify others' carts)
- Prepared statements (no SQL injection)

🎨 **User Experience**
- 8 products with images, names, prices
- Quantity selector before adding
- Cart summary with totals, shipping, tax
- Remove items without page refresh
- Loading states and error handling

---

## 📊 Database Schema

```sql
CREATE TABLE cart (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,          -- Links to users table
    product_id INT NOT NULL,        -- Product ID from shop
    product_name VARCHAR(255),      -- Product name
    price DECIMAL(10, 2),          -- Price per unit
    image VARCHAR(255),            -- Product image URL
    quantity INT DEFAULT 1,        -- Number of items
    created_at TIMESTAMP,          -- When added
    updated_at TIMESTAMP,          -- When updated
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

---

## 🧪 Testing Checklist

- [ ] Can login as customer (not admin)
- [ ] Can see 8 products on shop.php
- [ ] Click ADD TO CART → see green toast "✅ Product added to cart successfully"
- [ ] No page reload when adding items
- [ ] Can set quantity before adding (1-10)
- [ ] Can go to cart.php and see items
- [ ] Can remove items without page refresh
- [ ] Can see cart total, shipping, tax
- [ ] Empty cart shows "Your Cart is Empty" message
- [ ] Logout and login again → cart items persist

---

## 🐛 Troubleshooting

### Q: Can't add to cart
**A:** 
- Make sure you're logged in as CUSTOMER (not admin)
- Run migration: `http://localhost/interior/auth-system/migrate-cart.php`
- Check browser console (F12) for errors

### Q: Page reloads when clicking ADD TO CART
**A:**
- Check that JavaScript is enabled
- Check browser console for fetch() errors
- Make sure cart-api.php is returning JSON

### Q: Cart shows empty
**A:**
- Did you add items? Check cart-api.php response in console
- Log out and login again
- Check that product_id matches between shop.php and cart

### Q: Getting "Admin users cannot shop" message
**A:**
- You're logged in as ADMIN, not CUSTOMER
- Logout and register as CUSTOMER
- When registering, make sure Role = "CUSTOMER"

---

## 📱 API Endpoints

### GET /auth-system/cart-api.php?action=get
Returns all cart items for logged-in user

**Response:**
```json
{
  "success": true,
  "items": [
    {
      "id": 1,
      "product_id": 1,
      "product_name": "Nordic Wooden Chair",
      "price": 10000,
      "quantity": 2,
      "image": "assets/img/product/product_1_1.jpg"
    }
  ],
  "subtotal": 20000
}
```

### POST /auth-system/cart-api.php?action=add
Adds product to cart

**Parameters:**
- `product_id` - Product ID
- `product_name` - Name
- `price` - Price
- `image` - Image URL
- `quantity` - Quantity (default 1)

**Response:**
```json
{
  "success": true,
  "message": "Product added to cart successfully",
  "cart_count": 3
}
```

### POST /auth-system/cart-api.php?action=remove
Removes item from cart

**Parameters:**
- `cart_id` - Cart item ID

**Response:**
```json
{
  "success": true,
  "message": "Item removed from cart"
}
```

---

## 🚀 Next Steps

1. ✅ Test the system with these instructions
2. ✅ Customize product images and prices
3. ✅ Add more products by editing shop.php
4. ✅ Integrate with payment gateway (checkout.php)
5. ✅ Add order management system

---

**Created:** Real-Time Shopping Cart System
**Version:** 1.0
**Status:** Production Ready ✅
