/**
 * Global Authentication UI Manager
 * This script works on ALL pages to display user profile or login buttons
 * in the contact-strip area
 */

(function() {
  'use strict';

  // Function to get login status
  function isUserLoggedIn() {
    return localStorage.getItem('userLoggedIn') === 'true';
  }

  // Function to get user name
  function getUserName() {
    return localStorage.getItem('userName') || 'User';
  }

  // Function to initialize auth UI on page
  function initializeAuthUI() {
    // Check if contact-strip exists
    const contactStrip = document.querySelector('.contact-strip');
    if (!contactStrip) return; // Exit if no contact strip found

    // Check if auth elements already exist
    let authSection = document.getElementById('auth-section');
    let userProfileSection = document.getElementById('user-profile-section');

    // If they don't exist, create them
    if (!authSection) {
      // Create auth buttons section
      authSection = document.createElement('div');
      authSection.className = 'contact-item auth-buttons';
      authSection.id = 'auth-section';
      authSection.innerHTML = `
        <a href="auth-system/login.php" class="auth-btn login-btn">Login</a>
        <a href="auth-system/register.php" class="auth-btn register-btn">Register</a>
      `;
      contactStrip.appendChild(authSection);

      // Add auth button styles if not already present
      if (!document.getElementById('auth-styles')) {
        const style = document.createElement('style');
        style.id = 'auth-styles';
        style.textContent = `
          .auth-buttons {
            display: flex !important;
            gap: 12px;
            justify-content: center;
          }
          .auth-btn {
            padding: 10px 20px;
            border-radius: 4px;
            font-weight: 600;
            font-size: 14px;
            text-decoration: none;
            transition: all 0.3s ease;
            border: 2px solid #d4a33a;
          }
          .login-btn {
            background-color: #d4a33a;
            color: #ffffff;
          }
          .login-btn:hover {
            background-color: #ffffff;
            color: #d4a33a;
          }
          .register-btn {
            background-color: transparent;
            color: #d4a33a;
          }
          .register-btn:hover {
            background-color: #d4a33a;
            color: #ffffff;
          }
          .user-profile {
            gap: 15px !important;
          }
        `;
        document.head.appendChild(style);
      }
    }

    if (!userProfileSection) {
      // Create user profile section
      userProfileSection = document.createElement('div');
      userProfileSection.className = 'contact-item user-profile';
      userProfileSection.id = 'user-profile-section';
      userProfileSection.style.display = 'none';
      userProfileSection.innerHTML = `
        <div style="display: flex; align-items: center; gap: 10px;">
          <div style="width: 45px; height: 45px; background: #d4a33a; border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold;">
            <span id="user-avatar">U</span>
          </div>
          <div>
            <p class="contact-title" style="margin-bottom: 2px;">Welcome</p>
            <p class="contact-text" id="user-name-display">User</p>
          </div>
        </div>
        <a href="auth-system/logout.php" class="auth-btn login-btn" style="padding: 8px 15px; font-size: 13px;">Logout</a>
      `;
      contactStrip.appendChild(userProfileSection);
    }

    // Update UI based on login status
    updateAuthUI();
  }

  // Function to update auth UI display
  function updateAuthUI() {
    const authSection = document.getElementById('auth-section');
    const userProfileSection = document.getElementById('user-profile-section');

    if (!authSection || !userProfileSection) return;

    const isLoggedIn = isUserLoggedIn();

    if (isLoggedIn) {
      authSection.style.display = 'none';
      userProfileSection.style.display = 'flex';
      
      const userName = getUserName();
      const userNameDisplay = document.getElementById('user-name-display');
      const userAvatar = document.getElementById('user-avatar');
      
      if (userNameDisplay) userNameDisplay.textContent = userName;
      if (userAvatar) userAvatar.textContent = userName.charAt(0).toUpperCase();
    } else {
      authSection.style.display = 'flex';
      userProfileSection.style.display = 'none';
    }
  }

  // Initialize when DOM is ready
  function ready() {
    initializeAuthUI();
    
    // Listen for storage changes (from other tabs/windows)
    window.addEventListener('storage', function(e) {
      if (e.key === 'userLoggedIn' || e.key === 'userName') {
        updateAuthUI();
      }
    });

    // Also update when page becomes visible
    document.addEventListener('visibilitychange', function() {
      if (document.visibilityState === 'visible') {
        updateAuthUI();
      }
    });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', ready);
  } else {
    ready();
  }
})();
