# ⚡ Real-Time Verification Configuration Guide

## Complete Setup & Implementation

---

## 📦 What's Included

### Files Created
- ✅ `api-realtime-verify.php` - Main API endpoint (all verification logic)
- ✅ `realtime-verification.js` - JavaScript library for frontend
- ✅ `realtime-verification.css` - Beautiful UI styles
- ✅ `config-realtime-verify.php` - Configuration settings
- ✅ `realtime-verification-example.html` - Live example & documentation

---

## 🚀 Quick Start

### 1. Test It Now
Visit: **http://localhost/interior/auth-system/realtime-verification-example.html**

### 2. See Live Examples
- Test email availability check
- Test password strength validation
- Test phone number validation
- All with real-time feedback!

---

## 🔧 Configuration Settings

Edit `config-realtime-verify.php` to customize:

### Email Verification
```php
'email_verification' => array(
    'enabled' => true,
    'check_duplicate' => true,
    'blocked_domains' => array('tempmail.com', 'guerrillamail.com'),
    'dns_check' => false,
),
```

### OTP Settings
```php
'otp' => array(
    'length' => 6,              // 6-digit codes
    'expiry_minutes' => 10,     // Valid for 10 minutes
    'max_attempts' => 5,        // 5 failed attempts max
    'lock_minutes' => 15,       // Lock for 15 minutes after max attempts
    'resend_cooldown' => 60,    // Wait 60 seconds between resends
),
```

### Password Requirements
```php
'password' => array(
    'min_length' => 8,
    'require_uppercase' => true,
    'require_lowercase' => true,
    'require_numbers' => true,
    'require_special' => false,
    'special_chars' => '!@#$%^&*',
),
```

### Phone Validation
```php
'phone' => array(
    'enabled' => true,
    'min_length' => 10,
    'max_length' => 15,
    'allow_plus' => true,
    'allow_spaces' => true,
    'allow_hyphens' => true,
),
```

### 2FA Configuration
```php
'two_factor_auth' => array(
    'enabled' => true,
    'optional' => true,        // Users can enable/disable
    'default_enabled' => false,
    'methods' => array('email', 'sms'),
    'email_enabled' => true,
    'sms_enabled' => false,
),
```

### Security
```php
'security' => array(
    'require_email_verification' => true,
    'password_hash_algo' => PASSWORD_BCRYPT,
    'password_hash_cost' => 10,
    'https_only' => false,
    'csrf_protection' => true,
    'sql_injection_prevention' => true,
    'xss_prevention' => true,
),
```

---

## 💻 JavaScript Usage

### Include in Your HTML
```html
<!-- Styles -->
<link rel="stylesheet" href="realtime-verification.css">

<!-- JavaScript -->
<script src="realtime-verification.js"></script>
```

### Initialize on Form
```javascript
document.addEventListener('DOMContentLoaded', function() {
    RealtimeVerification.init('#myForm', {
        debounceDelay: 500,
        showFeedback: true,
        validatePassword: true
    });
});
```

### Manual Usage Examples

#### Check Email Availability
```javascript
RealtimeVerification.checkEmail('user@example.com', function(data) {
    if (data.available) {
        console.log('Email is available');
    } else {
        console.log('Email already exists');
    }
});
```

#### Check Password Strength
```javascript
RealtimeVerification.checkPasswordStrength('Test123!', function(data) {
    console.log('Strength:', data.strength_text);  // Weak, Fair, Good, Strong
    console.log('Feedback:', data.feedback);       // Array of improvement tips
    console.log('Valid:', data.valid);             // true/false
});
```

#### Send OTP
```javascript
RealtimeVerification.sendOTP('user@example.com', 'John Doe', function(data) {
    if (data.otp_sent) {
        console.log('OTP sent successfully');
    }
});
```

#### Verify OTP
```javascript
RealtimeVerification.verifyOTP('user@example.com', '123456', function(data) {
    if (data.verified) {
        console.log('OTP verified');
    } else {
        console.log('Invalid OTP');
    }
});
```

#### Check Password Match
```javascript
RealtimeVerification.checkPasswordMatch('Test123!', 'Test123!', function(data) {
    if (data.match) {
        console.log('Passwords match');
    }
});
```

#### Validate Phone
```javascript
RealtimeVerification.checkPhone('1234567890', function(data) {
    if (data.valid) {
        console.log('Phone valid:', data.formatted);
    }
});
```

#### Check 2FA Status
```javascript
RealtimeVerification.check2FA('user@example.com', function(data) {
    if (data.two_factor_enabled) {
        console.log('2FA is enabled');
    }
});
```

#### Start OTP Countdown Timer
```javascript
RealtimeVerification.startOTPTimer('#timerContainer', 'user@example.com', function() {
    console.log('OTP expired');
});
```

---

## 📡 API Endpoints

### Base URL
```
http://localhost/interior/auth-system/api-realtime-verify.php
```

### Endpoints

#### 1. Check Email Availability
```
GET ?action=check_email&email=user@example.com

Response:
{
    "status": "success",
    "available": true,
    "message": "Email is available"
}
```

#### 2. Check Password Strength
```
GET ?action=check_password&password=Test123!

Response:
{
    "status": "success",
    "strength": 4,
    "strength_text": "Good",
    "feedback": ["At least one special character"],
    "valid": true
}
```

#### 3. Check Password Match
```
GET ?action=check_password_match&password=Test123!&confirm=Test123!

Response:
{
    "status": "success",
    "match": true,
    "message": "Passwords match"
}
```

#### 4. Validate Phone
```
GET ?action=check_phone&phone=1234567890

Response:
{
    "status": "success",
    "valid": true,
    "message": "Phone number is valid",
    "formatted": "1234567890"
}
```

#### 5. Send OTP
```
GET ?action=send_otp&email=user@example.com&name=John%20Doe

Response:
{
    "status": "success",
    "otp_sent": true,
    "message": "OTP sent to your email"
}
```

#### 6. Verify OTP
```
GET ?action=verify_otp&email=user@example.com&otp=123456

Response:
{
    "status": "success",
    "verified": true,
    "message": "OTP verified successfully"
}
```

#### 7. Check Login Email
```
GET ?action=check_login_email&email=user@example.com&role=customer

Response:
{
    "status": "success",
    "exists": true,
    "message": "Email found"
}
```

#### 8. Check OTP Expiry
```
GET ?action=check_otp_expiry&email=user@example.com

Response:
{
    "status": "success",
    "expired": false,
    "expires_at": "2025-12-20 14:30:45",
    "remaining_seconds": 300,
    "remaining_minutes": 5
}
```

#### 9. Check 2FA Status
```
GET ?action=check_2fa&email=user@example.com

Response:
{
    "status": "success",
    "two_factor_enabled": false,
    "message": "2FA is not enabled"
}
```

---

## 🎯 Implementation Examples

### Example 1: Registration Form with Real-Time Verification

```html
<form id="registration-form" method="POST">
    <!-- Email Field -->
    <div class="form-group">
        <label>Email</label>
        <input type="email" name="email" required>
    </div>
    
    <!-- Password Field -->
    <div class="form-group">
        <label>Password</label>
        <input type="password" name="password" required>
    </div>
    
    <!-- Confirm Password -->
    <div class="form-group">
        <label>Confirm Password</label>
        <input type="password" name="confirm_password" required>
    </div>
    
    <!-- Phone Field -->
    <div class="form-group">
        <label>Phone</label>
        <input type="tel" name="phone" required>
    </div>
    
    <button type="submit">Register</button>
</form>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        RealtimeVerification.init('#registration-form');
    });
</script>
```

### Example 2: Manual Email Verification

```javascript
const emailInput = document.getElementById('emailInput');
const emailStatus = document.getElementById('emailStatus');

emailInput.addEventListener('blur', function() {
    RealtimeVerification.checkEmail(this.value, function(data) {
        if (data.available) {
            emailStatus.innerHTML = '✅ Available';
            emailStatus.style.color = 'green';
        } else {
            emailStatus.innerHTML = '❌ ' + data.message;
            emailStatus.style.color = 'red';
        }
    });
});
```

### Example 3: OTP Verification with Timer

```javascript
function sendOTP() {
    const email = document.getElementById('email').value;
    const name = document.getElementById('name').value;
    
    RealtimeVerification.sendOTP(email, name, function(data) {
        if (data.otp_sent) {
            document.getElementById('otpSection').style.display = 'block';
            
            // Start countdown timer
            RealtimeVerification.startOTPTimer('#otpTimer', email, function() {
                alert('OTP expired! Please request a new one.');
            });
        }
    });
}

function verifyOTP() {
    const email = document.getElementById('email').value;
    const otp = document.getElementById('otpCode').value;
    
    RealtimeVerification.verifyOTP(email, otp, function(data) {
        if (data.verified) {
            alert('OTP verified! Proceeding with registration...');
            document.getElementById('registrationForm').submit();
        } else {
            alert('Invalid OTP');
        }
    });
}
```

---

## 🔐 Security Features

### Built-In Protection
- ✅ Rate limiting (configurable requests per minute)
- ✅ Debounced API calls (prevent API spam)
- ✅ OTP time-limited (10 minutes default)
- ✅ Account lockout (after 5 failed attempts)
- ✅ SQL injection prevention (prepared statements)
- ✅ XSS protection (output escaping)
- ✅ CSRF protection ready
- ✅ IP tracking and logging

### Configuration
```php
'rate_limiting' => array(
    'enabled' => true,
    'check_email_rate' => 3,       // 3 per minute
    'send_otp_rate' => 2,          // 2 per minute
    'verify_otp_rate' => 10,       // 10 per minute
    'login_rate' => 5,             // 5 per minute
),
```

---

## 🧪 Testing Guide

### Test Email Check
```javascript
RealtimeVerification.checkEmail('test@example.com', function(data) {
    console.log(data);
});
```

### Test Password Strength
```javascript
RealtimeVerification.checkPasswordStrength('Test123!@', function(data) {
    console.log('Strength:', data.strength_text);
    console.log('Valid:', data.valid);
});
```

### Test OTP Flow
```javascript
// Send OTP
RealtimeVerification.sendOTP('test@example.com', 'Test User', function(data) {
    if (data.otp_sent) {
        // After user enters OTP...
        RealtimeVerification.verifyOTP('test@example.com', '123456', function(data) {
            console.log('Verified:', data.verified);
        });
    }
});
```

---

## 📊 Features Summary

| Feature | Status | Config File |
|---------|--------|-------------|
| Email verification | ✅ | `config-realtime-verify.php` |
| Password strength | ✅ | `config-realtime-verify.php` |
| Phone validation | ✅ | `config-realtime-verify.php` |
| OTP generation | ✅ | `config-realtime-verify.php` |
| 2FA detection | ✅ | `config-realtime-verify.php` |
| Rate limiting | ✅ | `config-realtime-verify.php` |
| Debouncing | ✅ | `realtime-verification.js` |
| Beautiful UI | ✅ | `realtime-verification.css` |
| Error handling | ✅ | `api-realtime-verify.php` |
| Logging | ✅ | `config-realtime-verify.php` |

---

## 📁 File Structure

```
auth-system/
├── api-realtime-verify.php              ← API endpoints
├── realtime-verification.js             ← JavaScript library
├── realtime-verification.css            ← Styles
├── config-realtime-verify.php           ← Configuration
├── realtime-verification-example.html   ← Live examples
├── register.php                         ← Your registration form
├── login.php                            ← Your login form
└── ... (other files)
```

---

## ✨ Integration Steps

### Step 1: Copy Files
Copy the 5 real-time verification files to your `auth-system/` folder

### Step 2: Update register.php
```html
<!-- Add to <head> -->
<link rel="stylesheet" href="realtime-verification.css">

<!-- Add before closing </body> -->
<script src="realtime-verification.js"></script>
<script>
    RealtimeVerification.init('#registrationForm');
</script>
```

### Step 3: Update login.php
Same as step 2, but initialize with login form ID

### Step 4: Customize Configuration
Edit `config-realtime-verify.php` to match your requirements

### Step 5: Test
Visit: http://localhost/interior/auth-system/realtime-verification-example.html

---

## 🎓 Learning Resources

- **Full Examples:** `realtime-verification-example.html`
- **Configuration:** `config-realtime-verify.php`
- **API Reference:** Read `api-realtime-verify.php` comments
- **JavaScript Docs:** Read `realtime-verification.js` comments

---

## 🐛 Troubleshooting

### Issue: API returning errors
**Solution:** Check `config-realtime-verify.php` is properly formatted

### Issue: Styles not showing
**Solution:** Verify CSS file path is correct in HTML

### Issue: OTP not sending
**Solution:** Check EmailService.php is configured correctly

### Issue: Rate limiting blocking requests
**Solution:** Adjust `rate_limiting` settings in config

---

## 📊 Status

✅ **Complete & Ready to Use**

- All 9 verification actions working
- Beautiful real-time UI
- Full configuration options
- Security features included
- Example implementation ready
- Live testing available

---

**Visit:** http://localhost/interior/auth-system/realtime-verification-example.html to test now! 🚀
