<?php
session_start();
require_once __DIR__ . '/config/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'customer') {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$page_title = 'My Profile — Interior Design';
$error = '';
$success = '';

// Get user info
$stmt = $conn->prepare('SELECT * FROM users WHERE id = ?');
$stmt->bind_param('i', $user_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    $name = trim($_POST['name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $address = trim($_POST['address'] ?? '');

    if (!$name) {
        $error = 'Name is required';
    } else {
        $stmt = $conn->prepare('UPDATE users SET name = ?, phone = ?, city = ?, address = ? WHERE id = ?');
        $stmt->bind_param('ssssi', $name, $phone, $city, $address, $user_id);

        if ($stmt->execute()) {
            $success = 'Profile updated successfully!';
            $_SESSION['user_name'] = $name;
            
            // Refresh user data
            $stmt = $conn->prepare('SELECT * FROM users WHERE id = ?');
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $user_result = $stmt->get_result();
            $user = $user_result->fetch_assoc();
        } else {
            $error = 'Failed to update profile';
        }
        $stmt->close();
    }
}

include 'header.php';
?>

<style>
    .dashboard-container {
        min-height: calc(100vh - 300px);
        background: linear-gradient(135deg, #f5f3f0 0%, #f9f7f4 100%);
        padding: 40px 0;
    }
    .dashboard-grid {
        display: grid;
        grid-template-columns: 280px 1fr;
        gap: 30px;
        max-width: 1400px;
        margin: 0 auto;
        padding: 0 20px;
    }
    .dashboard-sidebar {
        background: white;
        border-radius: 15px;
        padding: 30px 20px;
        height: fit-content;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        position: sticky;
        top: 100px;
    }
    .user-profile-mini {
        text-align: center;
        margin-bottom: 30px;
        padding-bottom: 20px;
        border-bottom: 2px solid #f0f0f0;
    }
    .user-avatar {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #D4A574 0%, #b8905a 100%);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 32px;
        color: white;
        margin: 0 auto 15px;
        box-shadow: 0 10px 25px rgba(212, 165, 116, 0.2);
    }
    .user-name {
        font-size: 18px;
        font-weight: 700;
        color: #1a1a1a;
        margin-bottom: 5px;
    }
    .user-email {
        font-size: 13px;
        color: #999;
    }
    .dashboard-menu {
        list-style: none;
        padding: 0;
        margin: 0;
    }
    .dashboard-menu a {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px 15px;
        color: #666;
        text-decoration: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 500;
        transition: all 0.3s ease;
    }
    .dashboard-menu a:hover,
    .dashboard-menu a.active {
        background: #f5f3f0;
        color: #D4A574;
    }
    .dashboard-menu i {
        width: 20px;
        text-align: center;
    }
    .dashboard-content {
        background: white;
        border-radius: 15px;
        padding: 40px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    }
    .dashboard-header {
        margin-bottom: 35px;
    }
    .dashboard-header h2 {
        font-size: 28px;
        font-weight: 700;
        color: #1a1a1a;
        margin: 0 0 10px 0;
    }
    .form-group {
        margin-bottom: 20px;
    }
    .form-group label {
        display: block;
        color: #333;
        font-size: 13px;
        font-weight: 600;
        margin-bottom: 7px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .form-control {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #e0e0e0;
        border-radius: 8px;
        font-size: 14px;
        font-family: 'DM Sans', sans-serif;
        transition: all 0.3s ease;
        background: #f9f9f9;
    }
    .form-control:focus {
        outline: none;
        border-color: #D4A574;
        background: white;
        box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
    }
    .form-row {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 20px;
    }
    .btn-save {
        background: linear-gradient(135deg, #D4A574 0%, #b8905a 100%);
        color: white;
        border: none;
        padding: 12px 30px;
        border-radius: 8px;
        font-weight: 700;
        font-size: 14px;
        cursor: pointer;
        transition: all 0.3s ease;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    .btn-save:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(212, 165, 116, 0.3);
    }
    .alert {
        padding: 14px 16px;
        border-radius: 8px;
        margin-bottom: 20px;
        border-left: 4px solid;
    }
    .alert-success {
        background-color: #f0fdf4;
        color: #155724;
        border-left-color: #28a745;
    }
    .alert-danger {
        background-color: #fff5f5;
        color: #721c24;
        border-left-color: #dc3545;
    }
    .info-box {
        background: #f5f3f0;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 25px;
        border-left: 4px solid #D4A574;
    }
    .info-box p {
        margin: 0;
        color: #666;
        font-size: 14px;
    }
    @media (max-width: 768px) {
        .dashboard-grid {
            grid-template-columns: 1fr;
        }
        .form-row {
            grid-template-columns: 1fr;
        }
        .dashboard-sidebar {
            position: relative;
            top: 0;
        }
    }
</style>

<div class="dashboard-container">
    <div class="dashboard-grid">
        <!-- Sidebar -->
        <div class="dashboard-sidebar">
            <div class="user-profile-mini">
                <div class="user-avatar">
                    <?php echo strtoupper(substr($user['name'], 0, 1)); ?>
                </div>
                <div class="user-name"><?php echo htmlspecialchars($user['name']); ?></div>
                <div class="user-email"><?php echo htmlspecialchars($user['email']); ?></div>
            </div>

            <ul class="dashboard-menu">
                <li><a href="customer-dashboard.php"><i class="fas fa-home"></i> Dashboard</a></li>
                <li><a href="profile.php" class="active"><i class="fas fa-user"></i> My Profile</a></li>
                <li><a href="settings.php"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="my-wishlist.php"><i class="fas fa-heart"></i> Wishlist</a></li>
                <li><a href="logout.php" onclick="return confirm('Are you sure you want to logout?');"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </div>

        <!-- Main Content -->
        <div class="dashboard-content">
            <div class="dashboard-header">
                <h2><i class="fas fa-user"></i> My Profile</h2>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>

            <div class="info-box">
                <p><i class="fas fa-info-circle"></i> Member since <?php echo date('M d, Y', strtotime($user['created_at'])); ?></p>
            </div>

            <form method="POST" action="">
                <div class="form-row">
                    <div class="form-group">
                        <label for="name"><i class="fas fa-user"></i> Full Name</label>
                        <input type="text" id="name" class="form-control" name="name" value="<?php echo htmlspecialchars($user['name']); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="email"><i class="fas fa-envelope"></i> Email Address</label>
                        <input type="email" class="form-control" value="<?php echo htmlspecialchars($user['email']); ?>" disabled>
                        <small style="color: #999; margin-top: 5px; display: block;">Cannot be changed</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="phone"><i class="fas fa-phone"></i> Phone Number</label>
                        <input type="tel" id="phone" class="form-control" name="phone" value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>" placeholder="Your phone number">
                    </div>
                    <div class="form-group">
                        <label for="city"><i class="fas fa-map-marker-alt"></i> City</label>
                        <input type="text" id="city" class="form-control" name="city" value="<?php echo htmlspecialchars($user['city'] ?? ''); ?>" placeholder="Your city">
                    </div>
                </div>

                <div class="form-group">
                    <label for="address"><i class="fas fa-home"></i> Address</label>
                    <input type="text" id="address" class="form-control" name="address" value="<?php echo htmlspecialchars($user['address'] ?? ''); ?>" placeholder="Your address">
                </div>

                <button type="submit" name="update_profile" class="btn-save">
                    <i class="fas fa-save"></i> Save Changes
                </button>
            </form>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>
