<?php
session_start();
require_once __DIR__ . '/config/db.php';
require_once __DIR__ . '/EmailService.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'customer') {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$page_title = 'Settings — Interior Design';
$error = '';
$success = '';
$email_service = new EmailService();

// Get user info
$stmt = $conn->prepare('SELECT * FROM users WHERE id = ?');
$stmt->bind_param('i', $user_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    if (!$current_password || !$new_password || !$confirm_password) {
        $error = 'All password fields are required';
    } elseif (!password_verify($current_password, $user['password'])) {
        $error = 'Current password is incorrect';
    } elseif (strlen($new_password) < 6) {
        $error = 'New password must be at least 6 characters';
    } elseif ($new_password !== $confirm_password) {
        $error = 'New passwords do not match';
    } else {
        $password_hash = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt = $conn->prepare('UPDATE users SET password = ?, password_changed_at = NOW() WHERE id = ?');
        $stmt->bind_param('si', $password_hash, $user_id);

        if ($stmt->execute()) {
            $success = 'Password changed successfully!';
        } else {
            $error = 'Failed to change password';
        }
        $stmt->close();
    }
}

// Handle 2FA toggle
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_2fa'])) {
    $enable_2fa = $_POST['enable_2fa'] ? 1 : 0;
    $stmt = $conn->prepare('UPDATE users SET two_factor_enabled = ? WHERE id = ?');
    $stmt->bind_param('ii', $enable_2fa, $user_id);

    if ($stmt->execute()) {
        $success = $enable_2fa ? '2FA enabled successfully!' : '2FA disabled successfully!';
        $user['two_factor_enabled'] = $enable_2fa;
    } else {
        $error = 'Failed to update 2FA settings';
    }
    $stmt->close();
}

include 'header.php';
?>

<style>
    .dashboard-container {
        min-height: calc(100vh - 300px);
        background: linear-gradient(135deg, #f5f3f0 0%, #f9f7f4 100%);
        padding: 40px 0;
    }
    .dashboard-grid {
        display: grid;
        grid-template-columns: 280px 1fr;
        gap: 30px;
        max-width: 1400px;
        margin: 0 auto;
        padding: 0 20px;
    }
    .dashboard-sidebar {
        background: white;
        border-radius: 15px;
        padding: 30px 20px;
        height: fit-content;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        position: sticky;
        top: 100px;
    }
    .user-profile-mini {
        text-align: center;
        margin-bottom: 30px;
        padding-bottom: 20px;
        border-bottom: 2px solid #f0f0f0;
    }
    .user-avatar {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #D4A574 0%, #b8905a 100%);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 32px;
        color: white;
        margin: 0 auto 15px;
        box-shadow: 0 10px 25px rgba(212, 165, 116, 0.2);
    }
    .user-name {
        font-size: 18px;
        font-weight: 700;
        color: #1a1a1a;
        margin-bottom: 5px;
    }
    .user-email {
        font-size: 13px;
        color: #999;
    }
    .dashboard-menu {
        list-style: none;
        padding: 0;
        margin: 0;
    }
    .dashboard-menu a {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px 15px;
        color: #666;
        text-decoration: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 500;
        transition: all 0.3s ease;
    }
    .dashboard-menu a:hover,
    .dashboard-menu a.active {
        background: #f5f3f0;
        color: #D4A574;
    }
    .dashboard-menu i {
        width: 20px;
        text-align: center;
    }
    .dashboard-content {
        background: white;
        border-radius: 15px;
        padding: 40px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    }
    .dashboard-header {
        margin-bottom: 35px;
    }
    .dashboard-header h2 {
        font-size: 28px;
        font-weight: 700;
        color: #1a1a1a;
        margin: 0 0 10px 0;
    }
    .settings-section {
        margin-bottom: 40px;
        padding-bottom: 30px;
        border-bottom: 2px solid #f0f0f0;
    }
    .settings-section:last-child {
        border-bottom: none;
    }
    .settings-title {
        font-size: 18px;
        font-weight: 700;
        color: #1a1a1a;
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }
    .form-group {
        margin-bottom: 20px;
    }
    .form-group label {
        display: block;
        color: #333;
        font-size: 13px;
        font-weight: 600;
        margin-bottom: 7px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .form-control {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #e0e0e0;
        border-radius: 8px;
        font-size: 14px;
        font-family: 'DM Sans', sans-serif;
        transition: all 0.3s ease;
        background: #f9f9f9;
    }
    .form-control:focus {
        outline: none;
        border-color: #D4A574;
        background: white;
        box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
    }
    .form-row {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 20px;
    }
    .toggle-switch {
        display: flex;
        align-items: center;
        gap: 15px;
    }
    .switch {
        position: relative;
        display: inline-block;
        width: 60px;
        height: 34px;
    }
    .switch input {
        opacity: 0;
        width: 0;
        height: 0;
    }
    .slider {
        position: absolute;
        cursor: pointer;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: #ccc;
        transition: .4s;
        border-radius: 34px;
    }
    .slider:before {
        position: absolute;
        content: "";
        height: 26px;
        width: 26px;
        left: 4px;
        bottom: 4px;
        background-color: white;
        transition: .4s;
        border-radius: 50%;
    }
    input:checked + .slider {
        background-color: #D4A574;
    }
    input:checked + .slider:before {
        transform: translateX(26px);
    }
    .btn-save {
        background: linear-gradient(135deg, #D4A574 0%, #b8905a 100%);
        color: white;
        border: none;
        padding: 12px 30px;
        border-radius: 8px;
        font-weight: 700;
        font-size: 14px;
        cursor: pointer;
        transition: all 0.3s ease;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    .btn-save:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(212, 165, 116, 0.3);
    }
    .alert {
        padding: 14px 16px;
        border-radius: 8px;
        margin-bottom: 20px;
        border-left: 4px solid;
    }
    .alert-success {
        background-color: #f0fdf4;
        color: #155724;
        border-left-color: #28a745;
    }
    .alert-danger {
        background-color: #fff5f5;
        color: #721c24;
        border-left-color: #dc3545;
    }
    .info-text {
        font-size: 13px;
        color: #999;
        margin-top: 5px;
    }
    @media (max-width: 768px) {
        .dashboard-grid {
            grid-template-columns: 1fr;
        }
        .form-row {
            grid-template-columns: 1fr;
        }
        .dashboard-sidebar {
            position: relative;
            top: 0;
        }
    }
</style>

<div class="dashboard-container">
    <div class="dashboard-grid">
        <!-- Sidebar -->
        <div class="dashboard-sidebar">
            <div class="user-profile-mini">
                <div class="user-avatar">
                    <?php echo strtoupper(substr($user['name'], 0, 1)); ?>
                </div>
                <div class="user-name"><?php echo htmlspecialchars($user['name']); ?></div>
                <div class="user-email"><?php echo htmlspecialchars($user['email']); ?></div>
            </div>

            <ul class="dashboard-menu">
                <li><a href="customer-dashboard.php"><i class="fas fa-home"></i> Dashboard</a></li>
                <li><a href="profile.php"><i class="fas fa-user"></i> My Profile</a></li>
                <li><a href="settings.php" class="active"><i class="fas fa-cog"></i> Settings</a></li>
                <li><a href="my-wishlist.php"><i class="fas fa-heart"></i> Wishlist</a></li>
                <li><a href="logout.php" onclick="return confirm('Are you sure you want to logout?');"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
            </ul>
        </div>

        <!-- Main Content -->
        <div class="dashboard-content">
            <div class="dashboard-header">
                <h2><i class="fas fa-cog"></i> Settings</h2>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>

            <!-- Change Password Section -->
            <div class="settings-section">
                <h3 class="settings-title">
                    <i class="fas fa-lock"></i> Change Password
                </h3>

                <form method="POST" action="">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="current_password"><i class="fas fa-lock"></i> Current Password</label>
                            <input type="password" id="current_password" class="form-control" name="current_password" placeholder="Enter current password" required>
                        </div>
                        <div class="form-group">
                            <label><i class="fas fa-info-circle"></i> Password Changed</label>
                            <input type="text" class="form-control" value="<?php echo date('M d, Y H:i', strtotime($user['password_changed_at'] ?? $user['created_at'])); ?>" disabled>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="new_password"><i class="fas fa-lock"></i> New Password</label>
                            <input type="password" id="new_password" class="form-control" name="new_password" placeholder="Min 6 characters" required>
                            <p class="info-text">At least 6 characters, mix of uppercase, lowercase, and numbers recommended</p>
                        </div>
                        <div class="form-group">
                            <label for="confirm_password"><i class="fas fa-lock"></i> Confirm New Password</label>
                            <input type="password" id="confirm_password" class="form-control" name="confirm_password" placeholder="Confirm password" required>
                        </div>
                    </div>

                    <button type="submit" name="change_password" class="btn-save">
                        <i class="fas fa-save"></i> Update Password
                    </button>
                </form>
            </div>

            <!-- 2FA Section -->
            <div class="settings-section">
                <h3 class="settings-title">
                    <i class="fas fa-shield-alt"></i> Two-Factor Authentication
                </h3>

                <form method="POST" action="">
                    <div class="toggle-switch">
                        <label class="switch">
                            <input type="checkbox" name="enable_2fa" value="1" <?php echo $user['two_factor_enabled'] ? 'checked' : ''; ?>>
                            <span class="slider"></span>
                        </label>
                        <div>
                            <p style="margin: 0 0 5px 0; font-weight: 600; color: #333;">Enable Two-Factor Authentication</p>
                            <p style="margin: 0; font-size: 13px; color: #999;">
                                <?php echo $user['two_factor_enabled'] ? 'You will be required to enter a code when logging in' : 'Protect your account with an additional security layer'; ?>
                            </p>
                        </div>
                    </div>

                    <button type="submit" name="toggle_2fa" class="btn-save" style="margin-top: 20px;">
                        <i class="fas fa-save"></i> Save 2FA Settings
                    </button>
                </form>
            </div>

            <!-- Account Status Section -->
            <div class="settings-section" style="border-bottom: none;">
                <h3 class="settings-title">
                    <i class="fas fa-info-circle"></i> Account Information
                </h3>

                <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 20px;">
                    <div>
                        <p style="font-size: 13px; color: #999; margin-bottom: 5px; text-transform: uppercase;">Account Status</p>
                        <p style="font-size: 16px; font-weight: 700; color: #D4A574;">Active</p>
                    </div>
                    <div>
                        <p style="font-size: 13px; color: #999; margin-bottom: 5px; text-transform: uppercase;">Email Verified</p>
                        <p style="font-size: 16px; font-weight: 700; color: #D4A574;">
                            <?php echo $user['email_verified'] ? '✓ Yes' : '✗ No'; ?>
                        </p>
                    </div>
                    <div>
                        <p style="font-size: 13px; color: #999; margin-bottom: 5px; text-transform: uppercase;">Member Since</p>
                        <p style="font-size: 14px; color: #333;"><?php echo date('M d, Y', strtotime($user['created_at'])); ?></p>
                    </div>
                    <div>
                        <p style="font-size: 13px; color: #999; margin-bottom: 5px; text-transform: uppercase;">Last Login</p>
                        <p style="font-size: 14px; color: #333;">View in dashboard</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>
