<?php
/**
 * User Registration Verification
 * Check if newly registered users are being saved correctly
 */

session_start();

if (!isset($_SESSION['user_id'])) {
    die('<h2>Please login first</h2>');
}

// Load database connection safely
$connection = null;
try {
    if (file_exists(__DIR__ . '/auth-system/config/db.php')) {
        require_once __DIR__ . '/auth-system/config/db.php';
    } else {
        die('<h2>Database configuration file not found</h2>');
    }
} catch (Exception $e) {
    die('<h2>Error loading database: ' . htmlspecialchars($e->getMessage()) . '</h2>');
}

if (!$connection || !($connection instanceof mysqli)) {
    die('<h2>Database connection failed. Please check your configuration.</h2>');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>User Registration Check</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; background: #f5f5f5; padding: 20px; }
        .container { max-width: 1000px; margin: 0 auto; }
        .card { background: white; padding: 25px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); margin: 20px 0; }
        h1 { color: #333; margin-bottom: 30px; }
        h2 { color: #333; border-bottom: 2px solid #d4a574; padding-bottom: 10px; margin-bottom: 20px; }
        table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #f9f9f9; font-weight: bold; }
        tr:hover { background: #f5f5f5; }
        .check { padding: 15px; margin: 15px 0; border-radius: 8px; border-left: 4px solid; }
        .pass { background: #d4edda; border-left-color: #28a745; }
        .fail { background: #f8d7da; border-left-color: #dc3545; }
        .info { background: #d1ecf1; border-left-color: #17a2b8; }
        .status { font-weight: bold; }
        .pass .status { color: #28a745; }
        .fail .status { color: #dc3545; }
        .code { background: #f4f4f4; padding: 15px; border-radius: 5px; font-family: monospace; font-size: 12px; overflow-x: auto; margin: 10px 0; }
        button { padding: 10px 20px; background: #d4a574; color: white; border: none; border-radius: 5px; cursor: pointer; font-weight: bold; margin-top: 10px; }
        button:hover { background: #b8905a; }
        .empty { color: #999; font-style: italic; }
        .footer { text-align: center; margin-top: 30px; }
        a { color: #d4a574; text-decoration: none; }
        a:hover { text-decoration: underline; }
    </style>
</head>
<body>
    <div class="container">
        <h1>👥 User Registration Verification</h1>

        <div class="card">
            <h2>✅ Users Table Structure</h2>
            <?php
            // Check table structure
            $result = $connection->query("DESCRIBE users");
            
            if ($result) {
                echo '<table>';
                echo '<thead><tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead>';
                echo '<tbody>';
                
                $columns = [];
                while ($row = $result->fetch_assoc()) {
                    $columns[] = $row['Field'];
                    echo '<tr>';
                    echo '<td><strong>' . htmlspecialchars($row['Field']) . '</strong></td>';
                    echo '<td>' . htmlspecialchars($row['Type']) . '</td>';
                    echo '<td>' . htmlspecialchars($row['Null']) . '</td>';
                    echo '<td>' . htmlspecialchars($row['Key']) . '</td>';
                    echo '<td>' . htmlspecialchars($row['Default'] ?? 'N/A') . '</td>';
                    echo '<td>' . htmlspecialchars($row['Extra']) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody>';
                echo '</table>';
                
                // Check for important columns
                echo '<div style="margin-top: 20px;">';
                $required = ['id', 'name', 'email', 'phone', 'password', 'created_at'];
                foreach ($required as $col) {
                    $has_col = in_array($col, $columns);
                    echo '<div class="check ' . ($has_col ? 'pass' : 'fail') . '">';
                    echo '<span class="status">' . ($has_col ? '✓' : '✗') . '</span> ';
                    echo 'Column "' . $col . '": ' . ($has_col ? 'Exists' : 'Missing');
                    echo '</div>';
                }
                echo '</div>';
            } else {
                echo '<div class="check fail"><span class="status">✗</span> Error reading table structure</div>';
            }
            ?>
        </div>

        <div class="card">
            <h2>📊 Registered Users Count</h2>
            <?php
            $result = $connection->query("SELECT COUNT(*) as count FROM users");
            $row = $result->fetch_assoc();
            $user_count = $row['count'];
            
            echo '<div class="check info">';
            echo '<strong>Total Users in Database:</strong> <span style="font-size: 24px; color: #d4a574; font-weight: bold;">' . $user_count . '</span>';
            echo '</div>';
            ?>
        </div>

        <div class="card">
            <h2>👤 Recently Registered Users</h2>
            <?php
            $result = $connection->query("SELECT id, name, email, phone, created_at FROM users ORDER BY created_at DESC LIMIT 10");
            
            if ($result && $result->num_rows > 0) {
                echo '<table>';
                echo '<thead><tr><th>ID</th><th>Name</th><th>Email</th><th>Phone</th><th>Registered</th></tr></thead>';
                echo '<tbody>';
                
                while ($row = $result->fetch_assoc()) {
                    echo '<tr>';
                    echo '<td><strong>#' . htmlspecialchars($row['id']) . '</strong></td>';
                    echo '<td>' . htmlspecialchars($row['name']) . '</td>';
                    echo '<td>' . htmlspecialchars($row['email']) . '</td>';
                    echo '<td>' . htmlspecialchars($row['phone'] ?? 'N/A') . '</td>';
                    echo '<td>' . date('M d, Y H:i', strtotime($row['created_at'])) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody>';
                echo '</table>';
            } else {
                echo '<p class="empty">No users found in database</p>';
            }
            ?>
        </div>

        <div class="card">
            <h2>🔍 Troubleshooting</h2>
            <div class="check info">
                <strong>If users are not showing in admin-dashboard:</strong><br>
                <ul style="margin: 10px 0 0 20px;">
                    <li>✓ Verify users are actually being registered (check this page)</li>
                    <li>✓ Check that admin-dashboard query is correct</li>
                    <li>✓ Make sure newly registered user data is displayed</li>
                </ul>
            </div>
            
            <div class="check info">
                <strong>If registration is failing:</strong><br>
                <ul style="margin: 10px 0 0 20px;">
                    <li>✓ Check that email sending works (test-email-config.php)</li>
                    <li>✓ Verify OTP is received in email</li>
                    <li>✓ Check browser console for errors</li>
                </ul>
            </div>
        </div>

        <div class="card">
            <h2>💡 Quick Links</h2>
            <p>
                <a href="admin-dashboard.php" style="display: inline-block; padding: 10px 15px; background: #d4a574; color: white; border-radius: 5px; margin-right: 10px;">Admin Dashboard</a>
                <a href="auth-system/register.php" style="display: inline-block; padding: 10px 15px; background: #6c757d; color: white; border-radius: 5px; margin-right: 10px;">Register New User</a>
                <a href="test-email-config.php" style="display: inline-block; padding: 10px 15px; background: #17a2b8; color: white; border-radius: 5px;">Test Email</a>
            </p>
        </div>

        <div class="footer">
            <p><a href="admin-dashboard.php">← Back to Dashboard</a></p>
        </div>
    </div>
</body>
</html>
