<?php
/**
 * Enhanced Email Service 
 * Uses DirectSMTPService for reliable email delivery
 * Works on Windows XAMPP without any external dependencies
 */

require_once __DIR__ . '/DirectSMTPService.php';

// Use DirectSMTPService as the main email service
class EmailService extends DirectSMTPService {}
?>

class EmailService {
    private $mail;
    private $smtp_host;
    private $smtp_port;
    private $smtp_user;
    private $smtp_pass;
    private $from_email;
    private $from_name;
    
    public function __construct() {
        $this->mail = new PHPMailer(true);
        
        // Load configuration from environment or config
        $this->smtp_host = getenv('SMTP_HOST') ?: 'mail.zemaraimgroups.com';
        $this->smtp_port = getenv('SMTP_PORT') ?: 465;
        $this->smtp_user = getenv('SMTP_USER') ?: 'info@zemaraimgroups.com';
        $this->smtp_pass = getenv('SMTP_PASS') ?: 'zemaraimgroups@123';
        $this->from_email = getenv('FROM_EMAIL') ?: 'info@zemaraimgroups.com';
        $this->from_name = 'Interior Design Solutions';
        
        $this->configureMailer();
    }
    
    private function configureMailer() {
        try {
            // SMTP Configuration
            $this->mail->isSMTP();
            $this->mail->Host = $this->smtp_host;
            $this->mail->SMTPAuth = true;
            $this->mail->Username = $this->smtp_user;
            $this->mail->Password = $this->smtp_pass;
            $this->mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            $this->mail->Port = $this->smtp_port;
            
            // Default settings
            $this->mail->setFrom($this->from_email, $this->from_name);
            $this->mail->isHTML(true);
            $this->mail->CharSet = 'UTF-8';
            
        } catch (Exception $e) {
            error_log("PHPMailer Configuration Error: " . $e->getMessage());
        }
    }
    
    /**
     * Send approval email
     */
    public function sendApprovalEmail($recipient_email, $name, $consultation_type, $preferred_date, $budget_range) {
        try {
            $this->mail->clearAddresses();
            $this->mail->addAddress($recipient_email);
            $this->mail->Subject = "Your Consultation Request Has Been Approved - Interior Design Solutions";
            
            $consultation_type = ucfirst(str_replace('_', ' ', $consultation_type));
            $preferred_date = date('F d, Y', strtotime($preferred_date));
            
            $message = '<html><head><style>';
            $message .= 'body { font-family: Segoe UI, Arial, sans-serif; line-height: 1.6; color: #333; }';
            $message .= '.container { max-width: 600px; margin: 0 auto; padding: 20px; background: #f9f7f4; }';
            $message .= '.header { background: linear-gradient(135deg, #8b6f47 0%, #6b5738 100%); color: white; padding: 30px; border-radius: 10px 10px 0 0; text-align: center; }';
            $message .= '.content { background: white; padding: 30px; border-radius: 0 0 10px 10px; }';
            $message .= '.approval-badge { background: #d4edda; color: #155724; padding: 15px; border-radius: 8px; margin: 20px 0; border: 1px solid #c3e6cb; text-align: center; font-weight: bold; }';
            $message .= '.details { background: #f9f7f4; padding: 20px; border-radius: 8px; margin: 20px 0; }';
            $message .= '.label { font-weight: 600; color: #8b6f47; }';
            $message .= '.next-steps { background: #e7f3ff; padding: 20px; border-radius: 8px; border-left: 4px solid #d4a574; }';
            $message .= '.footer { text-align: center; padding: 20px; color: #999; font-size: 12px; border-top: 1px solid #e0e0e0; margin-top: 20px; }';
            $message .= '</style></head><body>';
            $message .= '<div class="container">';
            $message .= '<div class="header"><h1 style="margin: 0; font-size: 28px;">Consultation Approved! ✅</h1></div>';
            $message .= '<div class="content">';
            $message .= '<p>Hello <strong>' . htmlspecialchars($name) . '</strong>,</p>';
            $message .= '<p>Great news! Your consultation request has been approved by our Interior Design Solutions team.</p>';
            $message .= '<div class="approval-badge">Your consultation has been approved</div>';
            $message .= '<h3 style="color: #8b6f47;">Your Consultation Details:</h3>';
            $message .= '<div class="details">';
            $message .= '<div><span class="label">Consultation Type:</span> ' . htmlspecialchars($consultation_type) . '</div>';
            $message .= '<div><span class="label">Preferred Date:</span> ' . htmlspecialchars($preferred_date) . '</div>';
            $message .= '<div><span class="label">Budget Range:</span> ' . htmlspecialchars($budget_range) . '</div>';
            $message .= '</div>';
            $message .= '<div class="next-steps">';
            $message .= '<h3 style="margin-top: 0; color: #0066cc;">What happens next?</h3>';
            $message .= '<ul style="line-height: 1.8;"><li>Our team will contact you within 24 hours</li>';
            $message .= '<li>We will discuss your project requirements in detail</li>';
            $message .= '<li>You will receive a customized proposal with timeline and pricing</li></ul>';
            $message .= '</div>';
            $message .= '<p>If you have any questions in the meantime, please don\'t hesitate to contact us.</p>';
            $message .= '<p>Best regards,<br><strong>Interior Design Solutions Team</strong><br><em>Creating spaces you\'ll love</em></p>';
            $message .= '<div class="footer">';
            $message .= '<p>This is an automated email. Please do not reply directly to this message.</p>';
            $message .= '<p>&copy; 2025 Interior Design Solutions. All rights reserved.</p>';
            $message .= '</div>';
            $message .= '</div></body></html>';
            
            $this->mail->Body = $message;
            
            if ($this->mail->send()) {
                error_log("✓ Approval email SENT via PHPMailer to: $recipient_email");
                return ['success' => true, 'message' => 'Email sent successfully'];
            }
            
        } catch (Exception $e) {
            error_log("✗ PHPMailer Error sending approval email: " . $this->mail->ErrorInfo);
            return ['success' => false, 'message' => $this->mail->ErrorInfo];
        }
    }
    
    /**
     * Send decline email
     */
    public function sendDeclineEmail($recipient_email, $name, $consultation_type, $preferred_date) {
        try {
            $this->mail->clearAddresses();
            $this->mail->addAddress($recipient_email);
            $this->mail->Subject = "Consultation Request Status Update - Interior Design Solutions";
            
            $consultation_type = ucfirst(str_replace('_', ' ', $consultation_type));
            $preferred_date = date('F d, Y', strtotime($preferred_date));
            
            $message = '<html><head><style>';
            $message .= 'body { font-family: Segoe UI, Arial, sans-serif; line-height: 1.6; color: #333; }';
            $message .= '.container { max-width: 600px; margin: 0 auto; padding: 20px; background: #f9f7f4; }';
            $message .= '.header { background: linear-gradient(135deg, #8b6f47 0%, #6b5738 100%); color: white; padding: 30px; border-radius: 10px 10px 0 0; text-align: center; }';
            $message .= '.content { background: white; padding: 30px; border-radius: 0 0 10px 10px; }';
            $message .= '.decline-badge { background: #fee2e2; color: #991b1b; padding: 15px; border-radius: 8px; margin: 20px 0; border: 1px solid #fecaca; text-align: center; font-weight: bold; font-size: 18px; }';
            $message .= '.details { background: #f9f7f4; padding: 20px; border-radius: 8px; margin: 20px 0; }';
            $message .= '.label { font-weight: 600; color: #8b6f47; }';
            $message .= '.next-steps { background: #fef3c7; padding: 20px; border-radius: 8px; border-left: 4px solid #d4a574; }';
            $message .= '.footer { text-align: center; padding: 20px; color: #999; font-size: 12px; border-top: 1px solid #e0e0e0; margin-top: 20px; }';
            $message .= '</style></head><body>';
            $message .= '<div class="container">';
            $message .= '<div class="header"><h1 style="margin: 0; font-size: 28px;">Consultation Request Update</h1></div>';
            $message .= '<div class="content">';
            $message .= '<p>Hello <strong>' . htmlspecialchars($name) . '</strong>,</p>';
            $message .= '<p>Thank you for submitting your consultation request with Interior Design Solutions.</p>';
            $message .= '<div class="decline-badge">❌ Request Not Approved at This Time</div>';
            $message .= '<h3 style="color: #8b6f47; margin-top: 30px;">Your Consultation Details:</h3>';
            $message .= '<div class="details">';
            $message .= '<div><span class="label">Consultation Type:</span> ' . htmlspecialchars($consultation_type) . '</div>';
            $message .= '<div><span class="label">Preferred Date:</span> ' . htmlspecialchars($preferred_date) . '</div>';
            $message .= '</div>';
            $message .= '<div class="next-steps">';
            $message .= '<h3 style="margin-top: 0; color: #8b6f47;">What you can do:</h3>';
            $message .= '<ul style="line-height: 1.8;"><li>Feel free to reach out to us directly if you have questions</li>';
            $message .= '<li>You can submit a new consultation request in the future</li>';
            $message .= '<li>Contact us to discuss alternative options or services</li>';
            $message .= '<li>Visit our portfolio to see our recent projects</li></ul>';
            $message .= '</div>';
            $message .= '<p>We appreciate your interest in our services and hope to work with you in the future. If you have any questions, please don\'t hesitate to contact us.</p>';
            $message .= '<p>Best regards,<br><strong>Interior Design Solutions Team</strong><br><em>Creating spaces you\'ll love</em></p>';
            $message .= '<div class="footer">';
            $message .= '<p>This is an automated email. Please do not reply directly to this message.</p>';
            $message .= '<p>&copy; 2025 Interior Design Solutions. All rights reserved.</p>';
            $message .= '</div>';
            $message .= '</div></body></html>';
            
            $this->mail->Body = $message;
            
            if ($this->mail->send()) {
                error_log("✓ Decline email SENT via PHPMailer to: $recipient_email");
                return ['success' => true, 'message' => 'Email sent successfully'];
            }
            
        } catch (Exception $e) {
            error_log("✗ PHPMailer Error sending decline email: " . $this->mail->ErrorInfo);
            return ['success' => false, 'message' => $this->mail->ErrorInfo];
        }
    }
    
    /**
     * Test SMTP connection
     */
    public function testConnection() {
        try {
            $this->mail->smtpConnect();
            return ['success' => true, 'message' => 'SMTP connection successful'];
        } catch (Exception $e) {
            error_log("✗ SMTP Connection Failed: " . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        } finally {
            $this->mail->smtpClose();
        }
    }
}
?>
