@echo off
REM SSL Certificate Fix for Composer
REM This script downloads and installs an updated CA certificate bundle

echo.
echo ========================================
echo SSL Certificate Bundle Update
echo ========================================
echo.

set CERT_DIR=C:\xampp\php
set CERT_FILE=%CERT_DIR%\cacert.pem
set BACKUP_FILE=%CERT_FILE%.backup

echo Current Certificate Location:
echo %CERT_FILE%
echo.

REM Check if PHP directory exists
if not exist "%CERT_DIR%" (
    echo ERROR: PHP directory not found: %CERT_DIR%
    pause
    exit /b 1
)

echo Downloading latest CA certificate bundle...
echo This may take a minute...
echo.

REM Download from Mozilla's trusted source
powershell -Command "
try {
    [Net.ServicePointManager]::SecurityProtocol = [Net.ServicePointManager]::SecurityProtocol -bor [Net.SecurityProtocolType]::Tls12
    
    $url = 'https://curl.se/ca/cacert.pem'
    $certPath = '%CERT_FILE%'
    
    Write-Host 'Downloading certificate from: '$url
    
    $webClient = New-Object System.Net.WebClient
    $webClient.DownloadFile($url, $certPath)
    
    Write-Host 'Certificate downloaded successfully!'
    Write-Host 'Location: '$certPath
    Write-Host 'Size: '+(Get-Item $certPath).Length+' bytes'
} catch {
    Write-Host 'ERROR: '$_.Exception.Message
    exit 1
}
"

if errorlevel 1 (
    echo.
    echo ERROR: Failed to download certificate!
    echo.
    echo Trying alternative download method...
    
    REM Alternative method using bitsadmin
    bitsadmin /transfer "CertDownload" "https://curl.se/ca/cacert.pem" "%CERT_FILE%"
    
    if errorlevel 1 (
        echo ERROR: All download methods failed
        pause
        exit /b 1
    )
)

echo.
echo ✓ Certificate bundle installed successfully!
echo.

REM Update php.ini to use the new certificate
echo Updating php.ini configuration...
echo.

powershell -NoProfile -ExecutionPolicy Bypass -Command "
try {
    $phpIniPath = 'C:\xampp\php\php.ini'
    $content = Get-Content $phpIniPath -Raw
    
    # Replace or add the openssl.cafile setting
    if ($content -match 'openssl\.cafile\s*=') {
        $content = $content -replace 'openssl\.cafile\s*=.*', 'openssl.cafile = %CERT_FILE%'
    } else {
        # Find the [openssl] section
        if ($content -match '\[openssl\]') {
            $content = $content -replace '(\[openssl\])', '[openssl]' + [System.Environment]::NewLine + 'openssl.cafile = %CERT_FILE%' + [System.Environment]::NewLine
        } else {
            # Add at the end if section doesn't exist
            $content += [System.Environment]::NewLine + '[openssl]' + [System.Environment]::NewLine + 'openssl.cafile = %CERT_FILE%' + [System.Environment]::NewLine
        }
    }
    
    # Also update curl.cainfo
    if ($content -match 'curl\.cainfo\s*=') {
        $content = $content -replace 'curl\.cainfo\s*=.*', 'curl.cainfo = %CERT_FILE%'
    } else {
        if ($content -match '\[curl\]') {
            $content = $content -replace '(\[curl\])', '[curl]' + [System.Environment]::NewLine + 'curl.cainfo = %CERT_FILE%' + [System.Environment]::NewLine
        }
    }
    
    $content | Set-Content $phpIniPath -Encoding UTF8
    Write-Host 'php.ini updated successfully!'
} catch {
    Write-Host 'WARNING: Could not auto-update php.ini'
    Write-Host 'You may need to manually add:'
    Write-Host 'openssl.cafile = C:\xampp\php\cacert.pem'
    Write-Host 'curl.cainfo = C:\xampp\php\cacert.pem'
}
"

echo.
echo ========================================
echo Installation Complete!
echo ========================================
echo.
echo Summary:
echo - Certificate: %CERT_FILE%
echo - php.ini: Updated with certificate path
echo.
echo Next: Run Composer
echo.
echo Command:
echo   cd C:\xampp\htdocs\interior
echo   composer require phpmailer/phpmailer
echo.

pause
