<!-- Test Dashboard & Verification -->
<?php
// test-dashboard.php - Comprehensive testing page for Customer Dashboard System

require_once 'config/db.php';

if (!isset($_SESSION['user_id'])) {
    session_start();
}

$tests = [];
$all_passed = true;

// Test 1: Database Connection
$test_name = "Database Connection";
try {
    $result = $connection->query("SELECT 1");
    $tests[$test_name] = ['status' => 'PASS', 'message' => 'Connected successfully'];
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => $e->getMessage()];
    $all_passed = false;
}

// Test 2: Users Table
$test_name = "Users Table";
try {
    $result = $connection->query("SHOW COLUMNS FROM users WHERE Field IN ('id', 'name', 'email', 'phone', 'address')");
    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    
    $required = ['id', 'name', 'email', 'phone', 'address'];
    $missing = array_diff($required, $columns);
    
    if (empty($missing)) {
        $tests[$test_name] = ['status' => 'PASS', 'message' => 'All required columns present'];
    } else {
        $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Missing columns: ' . implode(', ', $missing)];
        $all_passed = false;
    }
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => $e->getMessage()];
    $all_passed = false;
}

// Test 3: Orders Table
$test_name = "Orders Table";
try {
    $result = $connection->query("SHOW COLUMNS FROM orders WHERE Field IN ('id', 'user_id', 'status', 'total_amount', 'created_at')");
    $columns = [];
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    
    $required = ['id', 'user_id', 'status', 'total_amount', 'created_at'];
    $missing = array_diff($required, $columns);
    
    if (empty($missing)) {
        $tests[$test_name] = ['status' => 'PASS', 'message' => 'All required columns present'];
    } else {
        $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Missing columns: ' . implode(', ', $missing)];
        $all_passed = false;
    }
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => $e->getMessage()];
    $all_passed = false;
}

// Test 4: Consultations Table
$test_name = "Consultations Table";
try {
    $result = $connection->query("SHOW COLUMNS FROM consultations");
    if ($result && $result->num_rows > 0) {
        $count = $connection->query("SELECT COUNT(*) as count FROM consultations")->fetch_assoc()['count'];
        $tests[$test_name] = ['status' => 'PASS', 'message' => "Table exists with $count records"];
    } else {
        $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Table does not exist'];
        $all_passed = false;
    }
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Table does not exist. Run setup-consultations-db.php'];
    $all_passed = false;
}

// Test 5: API Files
$test_name = "API Files";
$api_files = [
    'api/update-profile.php',
    'api/submit-consultation.php'
];
$missing_files = [];
foreach ($api_files as $file) {
    if (!file_exists($file)) {
        $missing_files[] = $file;
    }
}

if (empty($missing_files)) {
    $tests[$test_name] = ['status' => 'PASS', 'message' => 'All API files present'];
} else {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Missing files: ' . implode(', ', $missing_files)];
    $all_passed = false;
}

// Test 6: Configuration Files
$test_name = "Configuration Files";
$config_files = [
    'config/db.php',
    'config/email.php'
];
$missing_files = [];
foreach ($config_files as $file) {
    if (!file_exists($file)) {
        $missing_files[] = $file;
    }
}

if (empty($missing_files)) {
    $tests[$test_name] = ['status' => 'PASS', 'message' => 'All config files present'];
} else {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Missing files: ' . implode(', ', $missing_files)];
    $all_passed = false;
}

// Test 7: Email Configuration
$test_name = "Email Configuration";
try {
    if (file_exists('config/email.php')) {
        $content = file_get_contents('config/email.php');
        if (strpos($content, 'MAIL_FROM_EMAIL') !== false && strpos($content, 'ADMIN_EMAIL') !== false) {
            $tests[$test_name] = ['status' => 'PASS', 'message' => 'Email config defines required constants'];
        } else {
            $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Email config missing required definitions'];
            $all_passed = false;
        }
    }
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => $e->getMessage()];
    $all_passed = false;
}

// Test 8: Page Files
$test_name = "Page Files";
$page_files = [
    'customer-dashboard.php',
    'consultation.php',
    'verify-consultation.php'
];
$missing_files = [];
foreach ($page_files as $file) {
    if (!file_exists($file)) {
        $missing_files[] = $file;
    }
}

if (empty($missing_files)) {
    $tests[$test_name] = ['status' => 'PASS', 'message' => 'All page files present'];
} else {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => 'Missing files: ' . implode(', ', $missing_files)];
    $all_passed = false;
}

// Test 9: Sample User
$test_name = "Sample User Data";
try {
    $result = $connection->query("SELECT COUNT(*) as count FROM users");
    $count = $result->fetch_assoc()['count'];
    if ($count > 0) {
        $tests[$test_name] = ['status' => 'PASS', 'message' => "Database has $count user(s)"];
    } else {
        $tests[$test_name] = ['status' => 'WARN', 'message' => 'No users in database. Register or add test user.'];
    }
} catch (Exception $e) {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => $e->getMessage()];
}

// Test 10: PHP Version
$test_name = "PHP Version";
if (version_compare(phpversion(), '7.4', '>=')) {
    $tests[$test_name] = ['status' => 'PASS', 'message' => 'PHP ' . phpversion() . ' (7.4+ required)'];
} else {
    $tests[$test_name] = ['status' => 'FAIL', 'message' => 'PHP ' . phpversion() . ' (need 7.4+)'];
    $all_passed = false;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Test - Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            padding: 40px 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }

        .status {
            display: inline-block;
            padding: 8px 16px;
            border-radius: 20px;
            font-weight: 600;
            margin-top: 15px;
        }

        .status.pass {
            background-color: rgba(39, 174, 96, 0.3);
            color: #27ae60;
        }

        .status.fail {
            background-color: rgba(231, 76, 60, 0.3);
            color: #e74c3c;
        }

        .tests {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .test-item {
            padding: 20px;
            border-bottom: 1px solid #e0e0e0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .test-item:last-child {
            border-bottom: none;
        }

        .test-item:hover {
            background-color: #f9f9f9;
        }

        .test-name {
            font-weight: 600;
            color: #333;
            flex: 1;
        }

        .test-badge {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 5px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            margin-right: 15px;
        }

        .test-badge.pass {
            background-color: #d4edda;
            color: #155724;
        }

        .test-badge.fail {
            background-color: #f8d7da;
            color: #721c24;
        }

        .test-badge.warn {
            background-color: #fff3cd;
            color: #856404;
        }

        .test-message {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }

        .next-steps {
            background: white;
            border-radius: 10px;
            padding: 30px;
            margin-top: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .next-steps h2 {
            color: #333;
            margin-bottom: 20px;
            font-size: 20px;
        }

        .next-steps ol {
            margin-left: 20px;
            color: #666;
        }

        .next-steps li {
            margin-bottom: 12px;
            line-height: 1.6;
        }

        .next-steps code {
            background-color: #f5f5f5;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
            color: #d63031;
        }

        .success-message {
            background-color: #d4edda;
            color: #155724;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            border: 1px solid #c3e6cb;
        }

        .error-message {
            background-color: #f8d7da;
            color: #721c24;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            border: 1px solid #f5c6cb;
        }

        @media (max-width: 600px) {
            .test-item {
                flex-direction: column;
                align-items: flex-start;
            }

            .test-badge {
                margin-right: 0;
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🧪 System Test</h1>
            <p>Customer Dashboard & Consultation System Health Check</p>
            <div class="status <?php echo $all_passed ? 'pass' : 'fail'; ?>">
                <?php echo $all_passed ? '✓ All Tests Passed' : '✗ Some Tests Failed'; ?>
            </div>
        </div>

        <?php if ($all_passed): ?>
            <div class="success-message">
                ✓ Your system is ready! All components are properly configured and the dashboard is ready to use.
            </div>
        <?php else: ?>
            <div class="error-message">
                ✗ Some tests failed. Please follow the instructions below to fix the issues before using the dashboard.
            </div>
        <?php endif; ?>

        <div class="tests">
            <?php foreach ($tests as $name => $test): ?>
                <div class="test-item">
                    <div style="flex: 1;">
                        <div class="test-name"><?php echo $name; ?></div>
                        <div class="test-message"><?php echo $test['message']; ?></div>
                    </div>
                    <span class="test-badge <?php echo strtolower($test['status']); ?>">
                        <?php echo $test['status']; ?>
                    </span>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="next-steps">
            <h2>Next Steps</h2>
            <ol>
                <li>
                    <strong>Create Database Table</strong><br>
                    Visit <code>http://localhost/interior/setup-consultations-db.php</code> or run the SQL migration
                </li>
                <li>
                    <strong>Configure Email Settings</strong><br>
                    Edit <code>config/email.php</code> and update MAIL_FROM_EMAIL and ADMIN_EMAIL
                </li>
                <li>
                    <strong>Add Phone & Address to Users</strong><br>
                    If not present, run: <code>ALTER TABLE users ADD COLUMN phone VARCHAR(20), address TEXT;</code>
                </li>
                <li>
                    <strong>Test the Dashboard</strong><br>
                    Visit <code>http://localhost/interior/customer-dashboard.php</code> (must be logged in)
                </li>
                <li>
                    <strong>Submit a Test Consultation</strong><br>
                    Go to the "Schedule Consultation" tab and fill out the form
                </li>
                <li>
                    <strong>Verify Email Link</strong><br>
                    Check your email inbox for the verification link and click it
                </li>
                <li>
                    <strong>Check Admin Panel</strong><br>
                    Visit <code>http://localhost/interior/admin-consultations.php</code> to see the consultation
                </li>
            </ol>
        </div>

        <div class="next-steps" style="margin-top: 30px;">
            <h2>📚 Documentation</h2>
            <ul style="margin-left: 20px; color: #666;">
                <li><strong>Quick Reference:</strong> <code>CUSTOMER_DASHBOARD_QUICK_REFERENCE.md</code></li>
                <li><strong>Complete Setup Guide:</strong> <code>CUSTOMER_DASHBOARD_SETUP.md</code></li>
                <li><strong>Implementation Summary:</strong> <code>CUSTOMER_DASHBOARD_IMPLEMENTATION_SUMMARY.md</code></li>
            </ul>
        </div>

        <div class="next-steps" style="margin-top: 30px; background-color: #f0f0f0;">
            <h2>🔗 Quick Links</h2>
            <ul style="margin-left: 20px; color: #666;">
                <li><a href="customer-dashboard.php" style="color: #667eea; text-decoration: none;">👤 Customer Dashboard</a></li>
                <li><a href="consultation.php" style="color: #667eea; text-decoration: none;">📋 Consultation Form</a></li>
                <li><a href="admin-consultations.php" style="color: #667eea; text-decoration: none;">⚙️ Admin Panel</a></li>
                <li><a href="setup-consultations-db.php" style="color: #667eea; text-decoration: none;">🗄️ Database Setup</a></li>
            </ul>
        </div>
    </div>
</body>
</html>
