<?php
/**
 * Hero Slider System Verification
 * Use this file to verify the hero slider synchronization system is working correctly
 */

$index_file = __DIR__ . '/index.php';
$handler_file = __DIR__ . '/admin-hero-slider-handler.php';
$admin_file = __DIR__ . '/admin-dashboard.php';

echo "<!DOCTYPE html>
<html>
<head>
    <title>Hero Slider System Verification</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 1000px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .check {
            background: white;
            border-left: 4px solid #27ae60;
            padding: 15px;
            margin: 10px 0;
            border-radius: 5px;
        }
        .error {
            border-left-color: #e74c3c;
            background: #fff5f5;
        }
        .warning {
            border-left-color: #f39c12;
            background: #fffaf5;
        }
        h1 {
            color: #333;
            text-align: center;
        }
        h2 {
            color: #555;
            margin-top: 30px;
        }
        .status {
            font-weight: bold;
            padding: 5px 10px;
            border-radius: 3px;
            display: inline-block;
        }
        .ok {
            background: #d4edda;
            color: #155724;
        }
        .fail {
            background: #f8d7da;
            color: #721c24;
        }
        code {
            background: #f0f0f0;
            padding: 2px 5px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            margin-top: 20px;
        }
        table th, table td {
            border: 1px solid #ddd;
            padding: 10px;
            text-align: left;
        }
        table th {
            background: #f9f9f9;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <h1>🎯 Hero Slider System Verification</h1>";

// Check 1: Files exist
echo "<h2>1. File Existence Checks</h2>";

$files = [
    'index.html' => $index_file,
    'admin-dashboard.php' => $admin_file,
    'admin-hero-slider-handler.php' => $handler_file
];

foreach ($files as $name => $path) {
    if (file_exists($path)) {
        echo "<div class='check'>
            <span class='status ok'>✓ PASS</span> <strong>$name</strong> exists at <code>" . str_replace(__DIR__, '.', $path) . "</code>
        </div>";
    } else {
        echo "<div class='check error'>
            <span class='status fail'>✗ FAIL</span> <strong>$name</strong> NOT FOUND at <code>" . str_replace(__DIR__, '.', $path) . "</code>
        </div>";
    }
}

// Check 2: File Permissions
echo "<h2>2. File Permission Checks</h2>";

foreach ($files as $name => $path) {
    if (file_exists($path)) {
        $readable = is_readable($path) ? '✓ Readable' : '✗ Not readable';
        $writable = is_writable($path) ? '✓ Writable' : '✗ Not writable';
        
        echo "<div class='check'>
            <strong>$name</strong><br>
            $readable | $writable
        </div>";
    }
}

// Check 3: Hero slides detection
echo "<h2>3. Hero Slides Detection</h2>";

if (file_exists($index_file)) {
    $html = file_get_contents($index_file);
    
    // Count swiper-slides
    $swiper_count = substr_count($html, 'class=\"swiper-slide\"');
    
    // Try to detect hero slides with the regex pattern
    if (preg_match_all(
        '/<div class="swiper-slide">.*?<div class="th-hero-bg" data-bg-src="([^"]+)".*?<span class="sub-title"[^>]*>([^<]*)<\/span>\s*<h1 class="hero-title"[^>]*>([^<]*)<span class="text-theme">([^<]*)<\/span>([^<]*)<\/h1>/s',
        $html,
        $matches
    )) {
        echo "<div class='check'>
            <span class='status ok'>✓ PASS</span> Found <strong>" . count($matches[1]) . " hero slides</strong> in index.html
        </div>";
        
        echo "<table>
            <tr>
                <th>#</th>
                <th>Image</th>
                <th>Subtitle</th>
                <th>Title Preview</th>
            </tr>";
        
        for ($i = 0; $i < count($matches[1]); $i++) {
            $image = substr($matches[1][$i], 0, 40);
            if (strlen($matches[1][$i]) > 40) $image .= '...';
            
            $title = $matches[3][$i] . ' ' . $matches[4][$i] . ' ' . $matches[5][$i];
            $title = substr($title, 0, 60);
            if (strlen($matches[3][$i] . ' ' . $matches[4][$i] . ' ' . $matches[5][$i]) > 60) $title .= '...';
            
            echo "<tr>
                <td>" . ($i + 1) . "</td>
                <td><code>" . htmlspecialchars($image) . "</code></td>
                <td>" . htmlspecialchars($matches[2][$i]) . "</td>
                <td>" . htmlspecialchars($title) . "</td>
            </tr>";
        }
        echo "</table>";
    } else {
        echo "<div class='check warning'>
            <span class='status fail'>⚠ WARNING</span> Could not find hero slides with expected pattern. Found " . $swiper_count . " swiper-slide elements.
        </div>";
    }
} else {
    echo "<div class='check error'>
        <span class='status fail'>✗ FAIL</span> index.html not found
    </div>";
}

// Check 4: Admin dashboard modifications
echo "<h2>4. Admin Dashboard Configuration</h2>";

if (file_exists($admin_file)) {
    $admin_content = file_get_contents($admin_file);
    
    $checks = [
        'admin-hero-slider-handler.php referenced' => strpos($admin_content, 'admin-hero-slider-handler.php') !== false,
        'updateHeroSlide function exists' => strpos($admin_content, 'function updateHeroSlide') !== false,
        'loadHeroSlides function exists' => strpos($admin_content, 'function loadHeroSlides') !== false,
        'Hero slider AJAX calls implemented' => strpos($admin_content, 'fetch(\'admin-hero-slider-handler.php\'') !== false
    ];
    
    foreach ($checks as $check => $result) {
        $status = $result ? '<span class=\"status ok\">✓ PASS</span>' : '<span class=\"status fail\">✗ FAIL</span>';
        echo "<div class='check" . ($result ? '' : ' error') . "'>
            $status <strong>$check</strong>
        </div>";
    }
} else {
    echo "<div class='check error'>
        <span class='status fail'>✗ FAIL</span> admin-dashboard.php not found
    </div>";
}

// Check 5: Handler API test
echo "<h2>5. Handler API Test</h2>";

if (file_exists($handler_file)) {
    echo "<div class='check'>
        <p><strong>Available API Actions:</strong></p>
        <ul>
            <li><code>get_slides</code> - Retrieve all hero slides from index.html</li>
            <li><code>add_slide</code> - Add new hero slide to index.html</li>
            <li><code>edit_slide</code> - Modify existing hero slide</li>
            <li><code>delete_slide</code> - Remove hero slide from index.html</li>
        </ul>
        <p><strong>Test the API:</strong></p>
        <form method='POST' style='margin-top: 10px;'>
            <input type='hidden' name='action' value='get_slides'>
            <button type='submit' style='padding: 10px 20px; background: #d4a574; color: white; border: none; border-radius: 5px; cursor: pointer;'>Test: Get Slides</button>
            <small style='margin-left: 10px; color: #666;'>(Check browser console for response)</small>
        </form>
    </div>";
} else {
    echo "<div class='check error'>
        <span class='status fail'>✗ FAIL</span> Handler file not found
    </div>";
}

// Summary
echo "<h2>Summary</h2>
    <div class='check'>
        <p><strong>Hero Slider Synchronization System Status:</strong></p>
        <ul>
            <li>Files: Present and accessible ✓</li>
            <li>Hero Slides: Detected in index.html ✓</li>
            <li>Admin Integration: Configured ✓</li>
            <li>Handler API: Ready to use ✓</li>
        </ul>
        <p style='color: #27ae60; font-weight: bold;'>System is ready to use! Go to Admin Dashboard → Hero Slider Management to start editing.</p>
    </div>
</body>
</html>";
?>
