# NextHM Authentication System Documentation

## Overview
The NextHM application implements a multi-role authentication system with role-based access control (RBAC). The system supports five user roles: Customer, Agent, Broker, Staff, and Admin.

## System Architecture

### Core Components

#### 1. **config.php** - Configuration & Database Connection
- Stores database credentials and configuration variables
- Establishes database connection using mysqli
- Location: `root/config.php`

#### 2. **login.php** - Login Interface
- HTML form for user authentication
- Collects username and password
- Form action: `process-login.php`
- Features:
  - Simple, clean login interface
  - Multi-role support
  - Error message display
  - Link to registration page

#### 3. **process-login.php** - Login Processing
- Validates user credentials
- Queries the `users` table
- Creates session variables:
  - `$_SESSION['user_id']` - Unique user identifier
  - `$_SESSION['user_name']` - User's full name
  - `$_SESSION['email']` - User's email
  - `$_SESSION['role']` - User's role (customer, agent, broker, staff, admin)
- Redirects to appropriate dashboard based on role
- Implements security measures:
  - SQL injection prevention with prepared statements
  - Password validation
  - Session initialization

#### 4. **logout.php** - Logout Handler
- Destroys user session
- Clears session variables
- Redirects to home page
- Location: `root/logout.php`

#### 5. **register.php** - Registration Interface
- HTML form for new user registration
- Collects user information (name, email, password, role)
- Form action: `process-register.php`
- Features:
  - Role selection dropdown
  - Input validation
  - Terms acceptance checkbox
  - Link to login page

#### 6. **process-register.php** - Registration Processing
- Validates registration form data
- Checks for duplicate emails
- Hashes password using `password_hash()`
- Inserts new user into `users` table
- Implements security:
  - Password strength validation
  - Email format validation
  - XSS prevention with htmlspecialchars()
  - CSRF token (optional)

### Dashboard System

#### Customer Dashboard (`dashboards/customer-dashboard.php`)
- **Access**: Customers only
- **Features**:
  - Saved properties statistics
  - Scheduled property visits
  - Inquiry tracking
  - Property browsing
- **Redirect**: Redirects to login if user is not a customer

#### Agent Dashboard (`dashboards/agent-dashboard.php`)
- **Access**: Agents only
- **Features**:
  - Total properties listing
  - Active inquiries management
  - Completed sales tracking
  - Commission earnings display
  - Recent properties table
  - Recent inquiries table
  - Quick action buttons

#### Broker Dashboard (`dashboards/broker-dashboard.php`)
- **Access**: Brokers only
- **Features**:
  - Agent management
  - Commission tracking
  - Market analysis
  - Team performance metrics

#### Staff Dashboard (`dashboards/staff-dashboard.php`)
- **Access**: Staff only
- **Features**:
  - Organization overview statistics
  - Management controls
  - Top performing agents table
  - Recent activities log
  - Agent and property management

#### Admin Dashboard (`dashboards/admin-dashboard.php`)
- **Access**: Admins only
- **Features**:
  - System administration
  - User management
  - Global statistics
  - System settings access

## Database Schema

### users Table

```sql
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    role ENUM('customer', 'agent', 'broker', 'staff', 'admin') NOT NULL,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

## Authentication Flow

### Login Process
```
1. User visits login.php
2. Enters credentials (username/email & password)
3. Form submits to process-login.php
4. process-login.php:
   - Validates credentials against users table
   - Verifies password with password_verify()
   - Creates session variables with user info
   - Redirects to appropriate dashboard
5. Dashboard checks $_SESSION variables
   - If valid: Displays dashboard
   - If invalid: Redirects to login.php
```

### Registration Process
```
1. User visits register.php
2. Fills registration form
3. Form submits to process-register.php
4. process-register.php:
   - Validates form data
   - Checks for duplicate email
   - Hashes password
   - Inserts user into database
   - Creates session or redirects to login
5. User redirected to login or logged in automatically
```

### Logout Process
```
1. User clicks Logout button
2. Navigates to logout.php
3. logout.php:
   - Starts session
   - Calls session_destroy()
   - Clears $_SESSION array
   - Redirects to home (index.php)
4. Session is destroyed, user is logged out
```

## Security Features

### 1. Password Security
- Passwords hashed using `password_hash()` with BCRYPT algorithm
- Password verification using `password_verify()`
- No plain text passwords stored

### 2. SQL Injection Prevention
- Prepared statements used for all database queries
- Parameter binding prevents SQL injection
- Example: `$stmt->bind_param("ss", $username, $password);`

### 3. Session Management
- Session tokens regenerated on login
- Session timeout implementation
- Session validation on each page load
- destroys session on logout

### 4. XSS Prevention
- User input escaped with `htmlspecialchars()`
- Output encoding prevents XSS attacks
- Example: `echo htmlspecialchars($user_name);`

### 5. CSRF Protection (Optional)
- Can implement CSRF tokens
- Verify tokens on form submissions
- Prevent cross-site request forgery

## Role-Based Access Control (RBAC)

Each dashboard implements role checking:

```php
<?php
session_start();

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'customer') {
    header('Location: ../login.php');
    exit;
}
// Dashboard content...
?>
```

### Role Hierarchy (Suggested)
1. **Admin**: Full system access
2. **Broker**: Agent management, reports
3. **Staff**: Operational management
4. **Agent**: Property management, inquiries
5. **Customer**: Browse properties, save favorites, schedule appointments

## API Endpoints

### `dashboards/customer-dashboard.php`
- **Method**: GET
- **Authentication**: Session required, role='customer'
- **Response**: HTML dashboard

### `dashboards/agent-dashboard.php`
- **Method**: GET
- **Authentication**: Session required, role='agent'
- **Response**: HTML dashboard

### `dashboards/staff-dashboard.php`
- **Method**: GET
- **Authentication**: Session required, role='staff'
- **Response**: HTML dashboard

### `process-login.php`
- **Method**: POST (from login.php)
- **Parameters**: `email/username`, `password`
- **Response**: Session creation + Redirect to dashboard

### `logout.php`
- **Method**: GET/POST
- **Parameters**: None
- **Response**: Session destruction + Redirect to home

## Testing the System

### Manual Testing

1. **Register a new user**:
   - Navigate to `register.php`
   - Fill in all fields
   - Submit form
   - Verify user created in database

2. **Login with credentials**:
   - Navigate to `login.php`
   - Enter test credentials
   - Click Login
   - Should redirect to appropriate dashboard

3. **Access dashboard directly without login**:
   - Try to access `dashboards/customer-dashboard.php` without session
   - Should redirect to `login.php`

4. **Logout**:
   - Click Logout button
   - Session should be destroyed
   - Redirected to home page

### Test Credentials (Examples)

**Customer Account**:
- Email: customer@test.com
- Password: CustomerPass123
- Role: customer

**Agent Account**:
- Email: agent@test.com
- Password: AgentPass123
- Role: agent

**Staff Account**:
- Email: staff@test.com
- Password: StaffPass123
- Role: staff

## File Structure

```
nexthm/
├── config.php                  # Database configuration
├── login.php                   # Login interface
├── process-login.php           # Login logic
├── logout.php                  # Logout handler
├── register.php                # Registration interface
├── process-register.php        # Registration logic
├── test-auth-system.php        # System test file
├── dashboards/
│   ├── customer-dashboard.php
│   ├── agent-dashboard.php
│   ├── staff-dashboard.php
│   ├── broker-dashboard.php
│   └── admin-dashboard.php
└── assets/
    ├── css/
    ├── js/
    └── img/
```

## Environment Variables (Recommended)

```php
// In config.php or .env file
DB_HOST = localhost
DB_USER = root
DB_PASSWORD = 
DB_NAME = nexthm
APP_URL = http://localhost/nexthm
SESSION_TIMEOUT = 3600  // 1 hour
```

## Future Enhancements

1. **Two-Factor Authentication (2FA)**
   - SMS/Email verification
   - Time-based OTP (TOTP)

2. **OAuth Integration**
   - Google Sign-in
   - Facebook Sign-in

3. **Advanced Session Management**
   - Remember me functionality
   - Session timeout warnings
   - Concurrent session limiting

4. **Audit Logging**
   - Track login attempts
   - Monitor sensitive actions
   - Generate audit reports

5. **Password Management**
   - Password reset via email
   - Password strength meter
   - Forced password change

6. **API Authentication**
   - JWT tokens
   - API key management
   - Rate limiting

## Troubleshooting

### Common Issues

1. **"Headers already sent" error**
   - Solution: Ensure `session_start()` is called before any output

2. **Session variables not persisting**
   - Solution: Check PHP session configuration
   - Verify session.save_path is writable

3. **Redirect loops**
   - Solution: Check role validation logic
   - Ensure correct redirect paths

4. **Database connection errors**
   - Solution: Verify credentials in config.php
   - Check database server is running

## Support & Maintenance

For issues or questions:
1. Check error logs in browser console
2. Review PHP error logs
3. Verify database schema matches documentation
4. Test with provided test credentials

---

**Last Updated**: February 2026
**Version**: 1.0
**Status**: Production Ready
