# Forgot Password & OTP System - Complete Implementation Guide

## Overview

A complete real-time password reset system with OTP (One-Time Password) verification using PHPMailer for email delivery. This system is fully responsive and works seamlessly across all devices.

## System Components

### 1. **auth-helpers.php**
Core helper functions for OTP generation, email sending, and password management.

**Key Functions:**
- `generateOTP()` - Generates a 6-digit random OTP
- `sendOTPEmail($email, $otp, $name)` - Sends OTP via email using PHPMailer
- `storeOTP($conn, $email, $otp)` - Stores OTP in database with 15-minute expiry
- `verifyOTP($conn, $email, $otp)` - Verifies OTP validity and expiration
- `resetPassword($conn, $email, $newPassword)` - Updates password in database
- `validatePassword($password)` - Validates password strength requirements
- `getUserByEmail($conn, $email)` - Retrieves user information

### 2. **Database Table: password_resets**
```sql
CREATE TABLE password_resets (
  id INT AUTO_INCREMENT PRIMARY KEY,
  email VARCHAR(100) NOT NULL,
  otp VARCHAR(6) NOT NULL,
  is_used BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  expires_at TIMESTAMP NULL,
  verified_at TIMESTAMP NULL,
  INDEX idx_email (email),
  INDEX idx_otp (otp)
)
```

### 3. **Pages**

#### **forgot-password.php**
- Entry point for password reset
- User enters registered email address
- Responsive grid layout with left sidebar info
- Form validation on client and server side
- Clean, modern design matching login page

#### **process-forgot-password.php**
- Handles form submission
- Generates random OTP
- Stores OTP in database
- Sends OTP via email using PHPMailer
- Redirects to verify-otp.php
- Security: Doesn't reveal if email exists

#### **verify-otp.php**
- Six individual OTP input fields
- Real-time input validation
- Auto-focus between fields
- Paste support for OTP code
- 15-minute countdown timer
- Resend OTP with 60-second cooldown
- Responsive design for all screens

#### **process-verify-otp.php**
- Validates OTP code
- Checks expiration time
- Confirms OTP hasn't been used
- Stores verification in session
- Redirects to reset-password.php

#### **reset-password.php**
- New password input with strength indicator
- Confirm password field
- Real-time password validation
- Shows requirements (length, uppercase, lowercase, number)
- Password match checker
- Toggle password visibility
- Submit button enabled only when all requirements met

#### **process-reset-password.php**
- Validates password strength
- Checks password confirmation
- Updates password in customers table
- Marks OTP as used
- Clears session variables
- Redirects to login with success message

#### **process-resend-otp.php**
- Allows users to request new OTP
- 60-second cooldown between sends
- Same email validation
- Prevents email enumeration attacks

## Configuration

### Email Configuration (auth-helpers.php)

Update these lines in `auth-helpers.php`:

```php
$mail->Host       = 'smtp.gmail.com'; // Your SMTP server
$mail->Username   = 'your-email@gmail.com'; // Your email
$mail->Password   = 'your-app-password';    // Your app password
$mail->setFrom('noreply@nexthm.com', 'NextHM Real Estate');
```

**For Gmail:**
1. Enable 2-Step Verification
2. Generate App Password from Google Account
3. Use the 16-character app password

**For Other Providers:**
- Update SMTP server and port
- Configure authentication method

## Security Features

1. **OTP Expiration**: 15-minute validity period
2. **One-Time Use**: OTP marked as used after verification
3. **Email Verification**: OTP sent only to registered email
4. **Password Strength**: Enforces 8+ chars, uppercase, lowercase, number
5. **Session Management**: Clears sensitive data after reset
6. **CSRF Protection**: Form-based submissions
7. **Rate Limiting**: 60-second cooldown on resend
8. **Input Validation**: Server-side validation of all inputs

## Password Requirements

- **Minimum Length**: 8 characters
- **Uppercase**: At least one (A-Z)
- **Lowercase**: At least one (a-z)
- **Numbers**: At least one (0-9)

## User Flow

```
1. User clicks "Forgot Password?" on login page
2. Enters registered email → process-forgot-password.php
3. OTP generated and sent to email
4. User enters OTP code on verify-otp.php
5. OTP verified → Redirects to reset-password.php
6. User sets new password with validation
7. Password updated in database
8. Redirected to login with success message
9. User can login with new password
```

## Responsive Design

All pages are fully responsive:
- **Desktop**: Grid layout with sidebar
- **Tablet (max-width: 992px)**: Single column layout
- **Mobile (max-width: 768px)**: Optimized spacing
- **Small Mobile (max-width: 480px)**: Minimal layout, sidebar hidden

## CSS Classes

### Common Styles
- `.alert` - Alert message container
- `.alert-danger` - Error messages
- `.alert-success` - Success messages
- `.alert-info` - Information messages
- `.form-group` - Form field container
- `.form-description` - Helper text below form fields

### Page-Specific Classes
- `.password-reset-section` - Main section wrapper
- `.otp-section` - OTP verification section
- `.password-strength` - Password strength indicator
- `.otp-input` - Individual OTP input field

## Testing Checklist

- [ ] User can access forgot-password.php
- [ ] Email validation works
- [ ] OTP is sent to correct email address
- [ ] OTP is 6 digits with format check
- [ ] OTP expires after 15 minutes
- [ ] OTP cannot be used twice
- [ ] Password strength validation works
- [ ] Resend OTP has 60-second cooldown
- [ ] Password reset updates database
- [ ] User can login with new password
- [ ] All pages are responsive on mobile
- [ ] Error messages display correctly
- [ ] Session is cleaned after reset

## Troubleshooting

### OTP not sending
1. Check email/password in auth-helpers.php
2. Verify SMTP server and port
3. Check firewall/hosting restrictions
4. Review error logs

### OTP expired too quickly
1. Check server time is accurate
2. Adjust INTERVAL 15 MINUTE in storeOTP()

### Password not updating
1. Verify database connection
2. Check column names match
3. Verify password_hash is working

## Integration with Login/Register

The forgot password system integrates with:
- [login.php](login.php) - Link on forgot password button
- [register.php](register.php) - Link for existing users
- [config.php](config.php) - Database connection and OTP table

## Future Enhancements

- SMS OTP delivery option
- Social login integration
- Two-factor authentication
- Biometric password reset
- Account recovery questions
- Backup codes for access

## Support

For issues or questions:
1. Check email configuration settings
2. Verify database table exists
3. Review server error logs
4. Test with different email providers
5. Verify password hash function works

## Files Summary

| File | Purpose |
|------|---------|
| auth-helpers.php | Helper functions for OTP and password |
| config.php | Database setup with password_resets table |
| forgot-password.php | Initial password reset page |
| process-forgot-password.php | Handle email submission |
| verify-otp.php | OTP verification page |
| process-verify-otp.php | Validate OTP |
| reset-password.php | Set new password page |
| process-reset-password.php | Update password |
| process-resend-otp.php | Resend OTP code |
| login.php | Updated with success message support |

---

**Version**: 1.0  
**Created**: February 2026  
**Status**: Production Ready
