# NextHM Quick Start Guide - Authentication System

## Installation & Setup

### Prerequisites
- XAMPP installed (Apache, PHP, MySQL)
- Database created (nexthm)
- Project files in `C:\xampp\htdocs\nexthm\`

### Step 1: Database Setup

Create the `users` table in MySQL:

```sql
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    role ENUM('customer', 'agent', 'broker', 'staff', 'admin') NOT NULL,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_role (role)
);
```

### Step 2: Create Test Users

Insert test users for each role:

```sql
INSERT INTO users (name, email, password, role) VALUES
('John Customer', 'customer@test.com', PASSWORD('CustomerPass123'), 'customer'),
('Jane Agent', 'agent@test.com', PASSWORD('AgentPass123'), 'agent'),
('Mike Staff', 'staff@test.com', PASSWORD('StaffPass123'), 'staff'),
('Bob Broker', 'broker@test.com', PASSWORD('BrokerPass123'), 'broker'),
('Admin User', 'admin@test.com', PASSWORD('AdminPass123'), 'admin');
```

**Note**: For production, use `password_hash()` instead of `PASSWORD()` function.

### Step 3: Verify Installation

1. Start XAMPP (Apache + MySQL)
2. Navigate to: `http://localhost/nexthm/test-auth-system.php`
3. Review system check results

## Quick Testing

### Test Login Flow

1. **Start at Login Page**:
   - Navigate to: `http://localhost/nexthm/login.php`

2. **Test Customer Login**:
   - Email: `customer@test.com`
   - Password: `CustomerPass123`
   - Expected: Redirects to Customer Dashboard

3. **Test Agent Login**:
   - Email: `agent@test.com`
   - Password: `AgentPass123`
   - Expected: Redirects to Agent Dashboard

4. **Test Staff Login**:
   - Email: `staff@test.com`
   - Password: `StaffPass123`
   - Expected: Redirects to Staff Dashboard

### Test Registration

1. Navigate to: `http://localhost/nexthm/register.php`
2. Fill in the form:
   - Full Name: Your Test Name
   - Email: your-test@email.com
   - Password: YourPassword123
   - Role: Select one (e.g., "Customer")
   - Accept Terms: Check box
3. Click Register
4. Should receive confirmation and redirect to login

### Test Access Control

1. **Login as Customer**
2. Try to access: `http://localhost/nexthm/dashboards/agent-dashboard.php`
   - Expected: Redirected to login.php (access denied)
3. Try to access: `http://localhost/nexthm/dashboards/customer-dashboard.php`
   - Expected: Shows customer dashboard

### Test Logout

1. Click "Logout" button in dashboard
2. Expected: Redirected to home page
3. Verify session is destroyed
4. Try to refresh dashboard:
   - Expected: Redirected to login.php

## Dashboard URLs

Once logged in, users can access their dashboards:

| Role | Dashboard URL |
|------|---------------|
| Customer | `/dashboards/customer-dashboard.php` |
| Agent | `/dashboards/agent-dashboard.php` |
| Staff | `/dashboards/staff-dashboard.php` |
| Broker | `/dashboards/broker-dashboard.php` |
| Admin | `/dashboards/admin-dashboard.php` |

## File Locations

```
C:\xampp\htdocs\nexthm\
├── login.php
├── register.php
├── logout.php
├── process-login.php
├── process-register.php
├── config.php
├── test-auth-system.php
└── dashboards/
    ├── customer-dashboard.php
    ├── agent-dashboard.php
    ├── staff-dashboard.php
    ├── broker-dashboard.php
    └── admin-dashboard.php
```

## Key Features

✅ Multi-role authentication system  
✅ Secure password hashing  
✅ SQL injection prevention  
✅ Session-based security  
✅ Role-based access control  
✅ Responsive dashboards  
✅ User registration  
✅ Logout functionality  

## Session Management

Sessions are stored in PHP's default session storage. To customize:

1. Edit `config.php`
2. Configure session parameters:
   ```php
   ini_set('session.gc_maxlifetime', 3600); // 1 hour
   ini_set('session.cookie_httponly', 1);   // Prevent XSS
   ini_set('session.cookie_secure', 1);     // HTTPS only (production)
   ```

## Debugging

### Enable Error Reporting (Development Only)

Add to `config.php`:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Check Session Variables

Add to any file:
```php
<?php
session_start();
echo '<pre>';
print_r($_SESSION);
echo '</pre>';
?>
```

## Security Checklist

Before going to production:

- [ ] Update database credentials in `config.php`
- [ ] Use `password_hash()` for all stored passwords
- [ ] Implement HTTPS (set `session.cookie_secure = 1`)
- [ ] Add CSRF token protection
- [ ] Implement rate limiting on login attempts
- [ ] Set appropriate file permissions (644 for files, 755 for directories)
- [ ] Disable error display in production
- [ ] Regular database backups
- [ ] Implement audit logging
- [ ] Test all access control rules

## Troubleshooting

### Problem: "Cannot modify header information"
**Solution**: Check for any output before `session_start()` or `header()` calls

### Problem: Session variables not working
**Solution**: 
1. Verify sessions folder is writable
2. Check `php.ini` session settings
3. Ensure `session_start()` is called first

### Problem: Login always fails
**Solution**:
1. Check database connection in `config.php`
2. Verify users table exists and has data
3. Check password hashing method

### Problem: Redirect loops
**Solution**: Review role validation logic in dashboard files

## Next Steps

1. **Customize Dashboards**: Edit dashboard files to add business logic
2. **Add More Features**: Implement properties, inquiries, etc.
3. **Setup Database Backup**: Configure automated backups
4. **SSL Certificate**: Install SSL for production
5. **Email Integration**: Add email notifications for registration/password reset

## Support Resources

- [PHP Sessions Documentation](https://www.php.net/manual/en/book.session.php)
- [Password Security](https://www.php.net/manual/en/function.password-hash.php)
- [XAMPP Documentation](https://www.apachefriends.org/)
- [MySQL Documentation](https://dev.mysql.com/doc/)

---

**Status**: Ready for Development  
**Last Updated**: February 2026  
**Version**: 1.0
