# NextHM - Real Estate Platform PHP Integration

## Overview

This is a complete PHP integration for the NextHM real estate platform with authentication, role-based login, customer registration, and database connectivity.

## ✅ Completed Features

### Authentication System
- ✅ Role-based login (Customer, Broker, Staff)
- ✅ Customer-only registration with email validation
- ✅ Password hashing using PHP bcrypt
- ✅ Session-based authentication
- ✅ Logout functionality
- ✅ Security with prepared statements (SQL injection prevention)

### Responsive Design
- ✅ Mobile-first approach (490px, 768px, 992px, 1920px breakpoints)
- ✅ Grid-based layouts for auth pages
- ✅ Responsive header with dynamic authentication display
- ✅ Responsive footer with newsletter form

### Pages & Functionality

#### Created PHP Files:

1. **config.php**
   - Database connection configuration
   - Session initialization
   - Base URL and root directory constants

2. **header.php**
   - Responsive navigation header (277 lines)
   - Session-based login/logout display
   - Dynamic user greeting with role display
   - Conditional rendering of auth buttons

3. **footer.php**
   - Responsive footer with contact information
   - Working hours, location, and contact details
   - Newsletter subscription form with email validation
   - Prepared statement for safe database insertion

4. **login.php**
   - Role-based account type selector (Customer/Broker/Staff)
   - Email and password fields
   - Prepared statement database query
   - Password verification using password_verify()
   - Session creation and redirect to dashboard

5. **register.php**
   - Customer-only registration form
   - Grid layout (2-column responsive design)
   - Password strength indicator with real-time feedback
   - Email validation and duplicate checking
   - Auto-login after successful registration
   - 6-character minimum password requirement

6. **logout.php**
   - Session destruction
   - Redirect to home page

7. **dashboard.php**
   - User profile display with role
   - Quick action buttons
   - Card-based dashboard layout
   - Links to properties, appointments, and settings

8. **setup.php**
   - One-click database table creation
   - Displays table creation status
   - Shows existing tables if already created

9. **index.php**
   - Home page with featured properties
   - Search functionality
   - Why Choose Us section
   - Testimonials
   - Call-to-action with dynamic buttons based on login status

## 🚀 Quick Start Guide

### Step 1: Access Database Setup
1. Open your browser and navigate to: `http://localhost/nexthm/setup.php`
2. Click **"Create Database Tables"** button
3. Wait for confirmation message

### Step 2: Register an Account
1. Go to: `http://localhost/nexthm/register.php`
2. Fill in your details (First Name, Last Name, Email, Phone, Password)
3. Check the password strength indicator
4. Accept terms and conditions
5. Click **"Create Account"**
6. You'll be automatically logged in and redirected to dashboard

### Step 3: Login with Different Roles
1. Go to: `http://localhost/nexthm/login.php`
2. Select account type (Customer/Broker/Staff)
3. Enter email and password
4. Click **"Sign In"**

**Demo Credentials** (after creating sample data):
- Customer: Use registered email
- Broker: Contact administrator
- Staff: Contact administrator

### Step 4: Browse the Platform
- Home: `http://localhost/nexthm/index.php`
- Dashboard: `http://localhost/nexthm/dashboard.php` (After login)
- Properties: `http://localhost/nexthm/properties.php`

## 📊 Database Schema

### Tables Created:

1. **customers**
   - id, firstname, lastname, email, phone, password, role, status, created_at, updated_at

2. **brokers**
   - id, name, email, phone, password, company, license_number, role, status, created_at, updated_at

3. **staff**
   - id, name, email, phone, password, position, role, status, created_at, updated_at

4. **properties**
   - id, title, description, price, type, status, address, city, state, zip_code, bedrooms, bathrooms, garage, area, broker_id, image_url, created_at, updated_at

5. **newsletter_subscribers**
   - id, email, subscribed_at (with UNIQUE constraint)

6. **appointments**
   - id, customer_id, property_id, broker_id, appointment_date, message, status, created_at, updated_at

7. **reviews**
   - id, customer_id, broker_id, property_id, rating (1-5), comment, created_at

8. **blog_posts**
   - id, title, slug, content, excerpt, featured_image, author_id, status, created_at, updated_at

## 🔒 Security Features

- **Password Hashing**: PHP password_hash() with BCRYPT algorithm
- **SQL Injection Prevention**: All queries use prepared statements with bind_param()
- **Email Validation**: filter_var() with FILTER_VALIDATE_EMAIL
- **XSS Prevention**: htmlspecialchars() on all output
- **Session Security**: Session variables properly initialized and validated
- **UNIQUE Constraints**: Email uniqueness enforced at database level

## 🎨 Styling & Responsive Design

### Color Scheme:
- Primary Gradient: #667eea to #764ba2
- Dark Background: #1a1a1a
- Light Background: #f8f9fa
- Accent Colors: Red (#ff4757), Green (#2ed573), Orange (#ffa502)

### Breakpoints:
- Mobile: 480px and below
- Tablet: 768px
- Desktop: 992px
- Large Desktop: 1920px

## 📝 File Structure

```
nexthm/
├── config.php                 # Database & session config
├── header.php                 # Responsive header include
├── footer.php                 # Responsive footer include
├── login.php                  # Login page (role-based)
├── register.php               # Registration page (customer)
├── logout.php                 # Logout handler
├── dashboard.php              # User dashboard
├── setup.php                  # Database setup wizard
├── index.php                  # Home page
├── login.html                 # Static login backup
├── register.html              # Static register backup
├── assets/
│   ├── css/
│   │   ├── style.css         # Main stylesheet
│   │   ├── bootstrap.min.css
│   │   ├── fontawesome.min.css
│   │   └── ... (other CSS files)
│   ├── js/
│   │   ├── main.js
│   │   ├── bootstrap.min.js
│   │   └── ... (other JS files)
│   └── img/                   # Images and icons
└── ... (other HTML pages)
```

## 🔧 Configuration

### Database Settings (config.php)
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'nexthm_db');
```

Modify these constants if your database credentials differ.

### Base URL
Update BASE_URL in config.php if deployed to a different location:
```php
define('BASE_URL', 'http://localhost/nexthm/');
```

## 📱 Responsive Breakpoints

| Device | Width | Layout |
|--------|-------|--------|
| Mobile | 480px | Single column |
| Tablet | 768px | 1-2 columns |
| Desktop | 992px | 2-3 columns |
| Large | 1920px | Full layout |

## 🚧 Next Steps (TODO)

- [ ] Convert properties.html → properties.php with database integration
- [ ] Convert about.html → about.php
- [ ] Convert service.html → service.php
- [ ] Convert appointment.html → appointment.php with booking logic
- [ ] Convert contact.html → contact.php with form submission
- [ ] Convert agents.html → agents.php with broker listings
- [ ] Convert blog.html → blog.php with blog posts from database
- [ ] Implement role-based dashboard pages
- [ ] Add password reset functionality
- [ ] Add admin panel for staff
- [ ] Implement email notifications
- [ ] Add payment gateway integration

## 🐛 Troubleshooting

### Database Connection Error
- Check XAMPP MySQL is running
- Verify database credentials in config.php
- Ensure nexthm_db database exists

### Login Not Working
- Verify tables were created via setup.php
- Check if user exists in the database
- Clear browser cookies/session

### Headers Already Sent Error
- Ensure no HTML output before require_once statements
- Check for BOM (Byte Order Mark) in files
- Verify no blank lines before PHP opening tag

### Newsletter Form Not Submitting
- Check POST method is used
- Verify form name matches in footer.php
- Check browser console for JavaScript errors

## 📧 Support

For issues or questions:
1. Check this README file
2. Review error messages in setup.php
3. Verify database tables were created
4. Clear session and try again

## 📄 License

This is a custom real estate platform. All rights reserved.

---

**Last Updated**: 2024
**Version**: 1.0
**PHP Version Required**: 5.6+
**MySQL Version Required**: 5.7+
