<?php
require_once 'config.php';

// Get property ID from URL
$property_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($property_id <= 0) {
    header('Location: properties.php');
    exit;
}

// Get property details
$query = "SELECT * FROM properties WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param('i', $property_id);
$stmt->execute();
$result = $stmt->get_result();
$property = $result->fetch_assoc();

if (!$property) {
    header('Location: properties.php');
    exit;
}

// Get broker info if exists
$broker = null;
if ($property['broker_id']) {
    $broker_query = "SELECT name, email, phone, company FROM brokers WHERE id = ?";
    $broker_stmt = $conn->prepare($broker_query);
    $broker_stmt->bind_param('i', $property['broker_id']);
    $broker_stmt->execute();
    $broker_result = $broker_stmt->get_result();
    $broker = $broker_result->fetch_assoc();
}

// Get reviews
$reviews_query = "SELECT r.*, c.firstname, c.lastname FROM reviews r JOIN customers c ON r.customer_id = c.id WHERE r.property_id = ? ORDER BY r.created_at DESC";
$reviews_stmt = $conn->prepare($reviews_query);
$reviews_stmt->bind_param('i', $property_id);
$reviews_stmt->execute();
$reviews_result = $reviews_stmt->get_result();
$reviews = [];
while ($review = $reviews_result->fetch_assoc()) {
    $reviews[] = $review;
}
?>
<!doctype html>
<html class="no-js" lang="zxx">

<head>
  <meta charset="utf-8">
  <meta http-equiv="x-ua-compatible" content="ie=edge">
  <title><?php echo htmlspecialchars($property['title']); ?> - Zemaraim</title>
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <link rel="icon" type="image/png" href="assets/img/favicons/favicon.png">
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin="">
  <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;500;600;700&family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="assets/css/bootstrap.min.css">
  <link rel="stylesheet" href="assets/css/fontawesome.min.css">
  <link rel="stylesheet" href="assets/css/magnific-popup.min.css">
  <link rel="stylesheet" href="assets/css/slick.min.css">
  <link rel="stylesheet" href="assets/css/style.css">
  <style>
    body { background: #f7f7f7; }
    .details-breadcrumb { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px 0; }
    .details-breadcrumb a, .details-breadcrumb span { color: white; }
    .details-breadcrumb a:hover { color: #ddd; text-decoration: none; }
    .separator { color: rgba(255,255,255,0.7); margin: 0 10px; }
    
    .image-gallery { position: relative; margin-bottom: 30px; border-radius: 10px; overflow: hidden; }
    .main-image { width: 100%; height: 400px; object-fit: cover; display: block; }
    .gallery-badge { position: absolute; top: 15px; right: 15px; background: #667eea; color: white; padding: 8px 15px; border-radius: 5px; font-size: 12px; font-weight: 600; }
    
    .property-info-card { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 15px rgba(0,0,0,0.08); margin-bottom: 30px; }
    .property-header { display: flex; justify-content: space-between; align-items: start; margin-bottom: 20px; flex-wrap: wrap; gap: 15px; }
    .property-title { font-size: 32px; font-weight: 700; color: #1a1a1a; margin: 0; }
    .property-status { display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 8px 15px; border-radius: 5px; font-size: 12px; font-weight: 600; text-transform: uppercase; }
    .property-price-section { text-align: right; }
    .property-large-price { font-size: 40px; font-weight: 700; color: #667eea; margin: 0px 0px 17px; }
    .property-address { color: #666; font-size: 14px; margin-top: 5px; }
        .features-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 52px; margin: 25px 0; }
    .feature-item { text-align: center; padding: 15px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #667eea; }
    .feature-icon { font-size: 24px; color: #667eea; margin-bottom: 8px; display: block; }
    .feature-label { font-size: 12px; color: #999; font-weight: 600; text-transform: uppercase; margin-bottom: 5px; }
    .feature-value { font-size: 18px; font-weight: 700; color: #1a1a1a; }
    
    .description-section, .broker-section, .reviews-section { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 15px rgba(0,0,0,0.08); margin-bottom: 30px; }
    .section-title { font-size: 22px; font-weight: 700; color: #1a1a1a; margin-bottom: 20px; border-bottom: 2px solid #667eea; padding-bottom: 10px; }
    .description-text { color: #555; line-height: 1.8; margin-bottom: 20px; }
    
    .broker-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 25px; border-radius: 10px; display: flex; gap: 20px; align-items: center; }
    .broker-avatar { width: 80px; height: 80px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 32px; flex-shrink: 0; }
    .broker-info h3 { margin: 0 0 5px 0; font-size: 18px; }
    .broker-contact { display: flex; gap: 15px; margin-top: 10px; flex-wrap: wrap; }
    .broker-contact a { color: white; text-decoration: none; display: flex; align-items: center; gap: 5px; padding: 6px 12px; background: rgba(255,255,255,0.2); border-radius: 5px; transition: all 0.3s ease; }
    .broker-contact a:hover { background: rgba(255,255,255,0.3); }
    
    .action-buttons { display: flex; gap: 10px; margin-top: 20px; flex-wrap: wrap; }
    .btn-action { padding: 12px 20px; border: none; border-radius: 5px; cursor: pointer; font-weight: 600; transition: all 0.3s ease; display: flex; align-items: center; gap: 8px; }
    .btn-schedule { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
    .btn-schedule:hover { transform: translateY(-2px); box-shadow: 0 5px 15px rgba(102,126,234,0.3); }
    .btn-schedule:disabled { opacity: 0.7; cursor: not-allowed; transform: none; }
    .btn-favorite { background: white; color: #667eea; border: 2px solid #667eea; }
    .btn-favorite:hover { background: #f8f9fa; }
    .btn-favorite.active { background: #ff4757; color: white; border-color: #ff4757; }
    
    .review-item { border-bottom: 1px solid #eee; padding: 20px 0; }
    .review-item:last-child { border-bottom: none; }
    .reviewer-name { font-weight: 700; color: #1a1a1a; margin-bottom: 5px; }
    .review-date { font-size: 12px; color: #999; margin-bottom: 10px; }
    .review-rating { color: #ffc107; margin-bottom: 10px; }
    .review-text { color: #555; line-height: 1.6; }
    
    .no-reviews { text-align: center; padding: 40px 20px; color: #999; }
    
    .form-control { padding: 10px 12px; border: 1px solid #e0e0e0; border-radius: 5px; font-size: 13px; width: 100%; transition: all 0.3s ease; }
    .form-control:focus { border-color: #667eea; box-shadow: 0 0 0 3px rgba(102,126,234,0.1); outline: none; }
    .form-group label { display: block; margin-bottom: 6px; font-weight: 600; font-size: 13px; color: #333; }
    
    /* Form message styles */
    #successMessage { animation: slideDown 0.3s ease-out; }
    #errorMessage { animation: slideDown 0.3s ease-out; }
    
    @keyframes slideDown {
      from {
        opacity: 0;
        transform: translateY(-10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }
    
    /* Tablet: 769px - 992px */
    @media (max-width: 992px) {
      .property-title { font-size: 26px; }
      .property-large-price { font-size: 34px; }
      .main-image { height: 320px; }
      .broker-card { flex-direction: column; text-align: center; }
      .property-header { flex-direction: column; gap: 20px; }
      .property-price-section { text-align: left; }
      .col-lg-4 { margin-top: 30px; }
    }
    
    /* Tablet: 641px - 768px */
    @media (max-width: 768px) {
      .main-image { height: 280px; }
      .property-info-card { padding: 20px; margin-bottom: 20px; }
      .description-section, .broker-section, .reviews-section { padding: 20px; margin-bottom: 20px; }
      .features-grid { grid-template-columns: repeat(3, 1fr); gap: 12px; }
      .feature-item { padding: 12px; border-left-width: 3px; }
      .feature-label { font-size: 11px; }
      .feature-value { font-size: 16px; }
      .property-title { font-size: 22px; }
      .property-large-price { font-size: 28px; }
      .section-title { font-size: 20px; }
      .action-buttons { gap: 10px; }
      .btn-action { padding: 11px 16px; font-size: 13px; }
      .contact-form-label { font-size: 12px; }
      .broker-contact a { font-size: 12px; }
    }
    
    /* Mobile: 481px - 640px */
    @media (max-width: 640px) {
      .main-image { height: 240px; }
      .property-info-card { padding: 15px; margin-bottom: 15px; }
      .description-section, .broker-section, .reviews-section { padding: 15px; margin-bottom: 15px; }
      .property-title { font-size: 20px; margin-bottom: 10px; }
      .property-large-price { font-size: 26px; }
      .features-grid { grid-template-columns: repeat(2, 1fr); gap: 10px; }
      .feature-item { padding: 10px; }
      .feature-icon { font-size: 20px; margin-bottom: 5px; }
      .feature-label { font-size: 10px; }
      .feature-value { font-size: 15px; }
      .section-title { font-size: 18px; margin-bottom: 15px; }
      .action-buttons { gap: 8px; flex-direction: column; }
      .btn-action { width: 100%; justify-content: center; padding: 10px 15px; }
      .broker-card { padding: 18px; gap: 12px; }
      .broker-avatar { width: 60px; height: 60px; font-size: 28px; }
      .broker-info h3 { font-size: 16px; }
      .broker-contact { gap: 8px; }
      .broker-contact a { font-size: 12px; padding: 5px 10px; }
    }
    
    /* Small Mobile: 0 - 480px */
    @media (max-width: 480px) {
      .main-image { height: 200px; }
      .property-info-card { padding: 12px; margin-bottom: 12px; }
      .description-section, .broker-section, .reviews-section { padding: 12px; margin-bottom: 12px; }
      .details-breadcrumb { padding: 20px 0; font-size: 12px; }
      .property-title { font-size: 18px; margin-bottom: 8px; }
      .property-large-price { font-size: 24px; }
      .property-address { font-size: 12px; }
      .features-grid { grid-template-columns: repeat(2, 1fr); gap: 8px; }
      .feature-item { padding: 8px; border-left-width: 3px; }
      .feature-icon { font-size: 18px; }
      .feature-label { font-size: 9px; }
      .feature-value { font-size: 13px; }
      .section-title { font-size: 16px; margin-bottom: 12px; padding-bottom: 8px; }
      .action-buttons { flex-direction: column; gap: 6px; }
      .btn-action { width: 100%; padding: 10px 12px; font-size: 12px; }
      .review-item { padding: 15px 0; }
      .reviewer-name { font-size: 14px; }
      .review-date { font-size: 11px; }
      .review-text { font-size: 13px; }
      .broker-card { padding: 15px; gap: 10px; }
      .broker-avatar { width: 50px; height: 50px; font-size: 24px; }
      .broker-info h3 { font-size: 14px; margin-bottom: 3px; }
      .broker-info p { font-size: 12px; }
      .broker-contact { flex-direction: column; }
      .broker-contact a { width: 100%; justify-content: center; font-size: 12px; }
    }
  </style>
</head>

<body>
  <?php include 'header.php'; ?>

  <!-- Breadcrumb -->
  <div class="details-breadcrumb">
    <div class="container">
      <a href="index.php"><i class="fas fa-home"></i> Home</a>
      <span class="separator">/</span>
      <a href="properties.php"><i class="fas fa-list"></i> Properties</a>
      <span class="separator">/</span>
      <span><?php echo htmlspecialchars($property['title']); ?></span>
    </div>
  </div>




  <!-- Property Details -->
  <section class="vs-section-padding" style="padding-top: 40px;">
    <div class="container">
      <div class="row g-4">
        
        <!-- Main Content -->
        <div class="col-lg-8">
          <!-- Image Gallery -->
          <div class="image-gallery">
            <img src="<?php echo !empty($property['image_url']) ? htmlspecialchars($property['image_url']) : 'assets/img/properties/properties-list-1.png'; ?>" alt="<?php echo htmlspecialchars($property['title']); ?>" class="main-image">
            <span class="gallery-badge"><i class="fas fa-images"></i> Featured Image</span>
          </div>

          <!-- Property Info -->
          <div class="property-info-card">
            <div class="property-header">
              <div class="property-title-section">
                <h1 class="property-title"><?php echo htmlspecialchars($property['title']); ?></h1>
                <div class="property-address">
                  <i class="fas fa-map-marker-alt" style="color: #667eea;"></i>
                  <?php echo htmlspecialchars($property['address']); ?>, <?php echo htmlspecialchars($property['city']); ?>, <?php echo htmlspecialchars($property['state']); ?> <?php echo htmlspecialchars($property['zip_code']); ?>
                </div>
              </div>
              <div class="property-price-section">
                <div class="property-large-price">$<?php echo number_format($property['price']); ?></div>
                <span class="property-status"><?php echo strtoupper($property['status']); ?></span>
              </div>
            </div>

            <!-- Features Grid -->
            <div class="features-grid">
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-bed"></i></span>
                <div class="feature-label">Bedrooms</div>
                <div class="feature-value"><?php echo $property['bedrooms']; ?></div>
              </div>
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-bath"></i></span>
                <div class="feature-label">Bathrooms</div>
                <div class="feature-value"><?php echo $property['bathrooms']; ?></div>
              </div>
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-car"></i></span>
                <div class="feature-label">Garage</div>
                <div class="feature-value"><?php echo $property['garage']; ?></div>
              </div>
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-ruler-combined"></i></span>
                <div class="feature-label">Area</div>
                <div class="feature-value"><?php echo number_format($property['area']); ?> sqft</div>
              </div>
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-tag"></i></span>
                <div class="feature-label">Type</div>
                <div class="feature-value" style="text-transform: capitalize;"><?php echo $property['type']; ?></div>
              </div>
              <div class="feature-item">
                <span class="feature-icon"><i class="fas fa-calendar-alt"></i></span>
                <div class="feature-label">Listed</div>
                <div class="feature-value"><?php echo date('M d, Y', strtotime($property['created_at'])); ?></div>
              </div>
            </div>

            <!-- Action Buttons -->
            <div class="action-buttons">
              <button class="btn-action btn-schedule" onclick="scheduleAppointment()">
                <i class="fas fa-calendar-check"></i> Schedule Appointment
              </button>
              <button class="btn-action btn-favorite" id="favoriteBtn" onclick="toggleFavorite()">
                <i class="fas fa-heart"></i> <span id="favText">Add to Favorites</span>
              </button>
            </div>
          </div>

          <!-- Description -->
          <div class="description-section">
            <h2 class="section-title"><i class="fas fa-align-left"></i> Description</h2>
            <p class="description-text"><?php echo htmlspecialchars($property['description']); ?></p>
          </div>

          <!-- Broker Info -->
          <?php if ($broker): ?>
            <div class="broker-section">
              <h2 class="section-title"><i class="fas fa-user-tie"></i> Broker Information</h2>
              <div class="broker-card">
                <div class="broker-avatar">
                  <i class="fas fa-user"></i>
                </div>
                <div class="broker-info">
                  <h3><?php echo htmlspecialchars($broker['name']); ?></h3>
                  <p style="margin: 5px 0; font-size: 14px;"><?php echo htmlspecialchars($broker['company']); ?></p>
                  <div class="broker-contact">
                    <a href="mailto:<?php echo htmlspecialchars($broker['email']); ?>">
                      <i class="fas fa-envelope"></i> Email
                    </a>
                    <a href="tel:<?php echo htmlspecialchars($broker['phone']); ?>">
                      <i class="fas fa-phone"></i> Call
                    </a>
                  </div>
                </div>
              </div>
            </div>
          <?php endif; ?>

          <!-- Reviews -->
          <div class="reviews-section">
            <h2 class="section-title"><i class="fas fa-star"></i> Reviews (<?php echo count($reviews); ?>)</h2>
            <?php if (!empty($reviews)): ?>
              <?php foreach ($reviews as $review): ?>
                <div class="review-item">
                  <div class="reviewer-name"><?php echo htmlspecialchars($review['firstname'] . ' ' . $review['lastname']); ?></div>
                  <div class="review-date"><?php echo date('M d, Y', strtotime($review['created_at'])); ?></div>
                  <div class="review-rating">
                    <?php for ($i = 0; $i < $review['rating']; $i++): ?>
                      <i class="fas fa-star"></i>
                    <?php endfor; ?>
                    <?php for ($i = $review['rating']; $i < 5; $i++): ?>
                      <i class="far fa-star"></i>
                    <?php endfor; ?>
                  </div>
                  <div class="review-text"><?php echo htmlspecialchars($review['comment']); ?></div>
                </div>
              <?php endforeach; ?>
            <?php else: ?>
              <div class="no-reviews">
                <i class="fas fa-comment-slash" style="font-size: 40px; color: #ddd;"></i>
                <p style="margin-top: 15px;">No reviews yet. Be the first to review this property!</p>
              </div>
            <?php endif; ?>
          </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
          <!-- Quick Contact Widget -->
          <div class="property-info-card">
            <h3 style="margin-top: 0; color: #1a1a1a;"><i class="fas fa-phone"></i> Quick Contact</h3>
            <form id="inquiryForm" onsubmit="handleContactSubmit(event)">
              <div class="form-group" style="margin-bottom: 15px;">
                <label for="inquiry_name">Full Name</label>
                <input type="text" id="inquiry_name" name="full_name" class="form-control" placeholder="Your name" required>
              </div>
              <div class="form-group" style="margin-bottom: 15px;">
                <label for="inquiry_email">Email</label>
                <input type="email" id="inquiry_email" name="email" class="form-control" placeholder="your@email.com" required>
              </div>
              <div class="form-group" style="margin-bottom: 15px;">
                <label for="inquiry_phone">Phone</label>
                <input type="tel" id="inquiry_phone" name="phone" class="form-control" placeholder="Your phone number" required>
              </div>
              <div class="form-group" style="margin-bottom: 15px;">
                <label for="inquiry_message">Message</label>
                <textarea id="inquiry_message" name="message" class="form-control" rows="4" placeholder="Your message..." style="resize: vertical;" required></textarea>
              </div>
              <button type="submit" id="submitBtn" class="btn-action btn-schedule" style="width: 100%;">
                <i class="fas fa-paper-plane"></i> Send Inquiry
              </button>
              <div id="successMessage" style="display: none; margin-top: 10px; padding: 12px; background: #dcfce7; border: 1px solid #86efac; color: #166534; border-radius: 5px; text-align: center;">
                <i class="fas fa-check-circle"></i> Inquiry sent successfully!
              </div>
              <div id="errorMessage" style="display: none; margin-top: 10px; padding: 12px; background: #fee2e2; border: 1px solid #fca5a5; color: #991b1b; border-radius: 5px; text-align: center;">
                <span id="errorText"></span>
              </div>
            </form>
          </div>

          <!-- Property Details List -->
          <div class="property-info-card" style="margin-top: 30px;">
            <h3 style="margin-top: 0; color: #1a1a1a;"><i class="fas fa-list-ul"></i> Details</h3>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; font-size: 13px;">
              <div>
                <strong style="color: #667eea;">Type:</strong><br>
                <span style="color: #666; text-transform: capitalize;"><?php echo $property['type']; ?></span>
              </div>
              <div>
                <strong style="color: #667eea;">Status:</strong><br>
                <span style="color: #666; text-transform: capitalize;"><?php echo $property['status']; ?></span>
              </div>
              <div>
                <strong style="color: #667eea;">Bedrooms:</strong><br>
                <span style="color: #666;"><?php echo $property['bedrooms']; ?></span>
              </div>
              <div>
                <strong style="color: #667eea;">Bathrooms:</strong><br>
                <span style="color: #666;"><?php echo $property['bathrooms']; ?></span>
              </div>
              <div>
                <strong style="color: #667eea;">Area:</strong><br>
                <span style="color: #666;"><?php echo number_format($property['area']); ?> sqft</span>
              </div>
              <div>
                <strong style="color: #667eea;">Garage:</strong><br>
                <span style="color: #666;"><?php echo $property['garage']; ?> spaces</span>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </section>

  <?php include 'footer.php'; ?>

  <script src="assets/js/vendor/jquery-3.6.0.min.js"></script>
  <script src="assets/js/bootstrap.min.js"></script>
  <script src="assets/js/main.js"></script>

  <script>
    function toggleFavorite() {
      const btn = document.getElementById('favoriteBtn');
      const favText = document.getElementById('favText');
      btn.classList.toggle('active');
      
      let favorites = JSON.parse(localStorage.getItem('favorites') || '[]');
      const propId = <?php echo $property_id; ?>;
      
      if (btn.classList.contains('active')) {
        if (!favorites.includes(propId)) favorites.push(propId);
        favText.textContent = 'Added to Favorites';
      } else {
        favorites = favorites.filter(id => id !== propId);
        favText.textContent = 'Add to Favorites';
      }
      localStorage.setItem('favorites', JSON.stringify(favorites));
    }

    function scheduleAppointment() {
      alert('Schedule appointment feature coming soon!');
    }

    async function handleContactSubmit(e) {
      e.preventDefault();
      
      const form = document.getElementById('inquiryForm');
      const submitBtn = document.getElementById('submitBtn');
      const successMessage = document.getElementById('successMessage');
      const errorMessage = document.getElementById('errorMessage');
      const errorText = document.getElementById('errorText');
      
      // Hide previous messages
      successMessage.style.display = 'none';
      errorMessage.style.display = 'none';
      
      // Disable button and show loading state
      const originalText = submitBtn.innerHTML;
      submitBtn.disabled = true;
      submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
      
      try {
        const formData = {
          full_name: document.getElementById('inquiry_name').value.trim(),
          email: document.getElementById('inquiry_email').value.trim(),
          phone: document.getElementById('inquiry_phone').value.trim(),
          message: document.getElementById('inquiry_message').value.trim(),
          property_id: <?php echo $property_id; ?>
        };
        
        const response = await fetch('api/save-inquiry.php', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json'
          },
          body: JSON.stringify(formData)
        });
        
        const data = await response.json();
        
        if (data.success) {
          successMessage.style.display = 'block';
          form.reset();
          
          // Revert button after 3 seconds
          setTimeout(() => {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalText;
            successMessage.style.display = 'none';
          }, 3000);
        } else {
          errorText.textContent = data.message || 'Failed to send inquiry. Please try again.';
          errorMessage.style.display = 'block';
          submitBtn.disabled = false;
          submitBtn.innerHTML = originalText;
        }
      } catch (error) {
        console.error('Error:', error);
        errorText.textContent = 'Error sending inquiry. Please try again.';
        errorMessage.style.display = 'block';
        submitBtn.disabled = false;
        submitBtn.innerHTML = originalText;
      }
    }

    // Load favorites on page load
    document.addEventListener('DOMContentLoaded', function() {
      const favorites = JSON.parse(localStorage.getItem('favorites') || '[]');
      const propId = <?php echo $property_id; ?>;
      if (favorites.includes(propId)) {
        document.getElementById('favoriteBtn').classList.add('active');
        document.getElementById('favText').textContent = 'Added to Favorites';
      }
    });
  </script>
</body>

</html>
  