<?php
/**
 * Zemaraim Database Setup - Create Test Users
 * Run this file once: http://localhost/Zemaraim/setup-users.php
 */

require_once 'config.php';

$setup_messages = [];

try {
    // Check if users table exists
    $check_table = $conn->query("SELECT COUNT(*) as count FROM users");
    
    if (!$check_table) {
        $setup_messages[] = "❌ Error: Users table doesn't exist. Please check config.php.";
    } else {
        // Clear existing test users
        $existing_count = $check_table->fetch_assoc()['count'];
        
        if ($existing_count > 0) {
            $conn->query("DELETE FROM users WHERE email IN ('admin@test.com', 'customer@test.com', 'agent@test.com', 'staff@test.com', 'broker@test.com')");
            $setup_messages[] = "✓ Cleared existing test users";
        }
        
        // Create test users with hashed passwords
        $test_users = [
            [
                'full_name' => 'Admin User',
                'email' => 'admin@test.com',
                'password' => password_hash('AdminPass123', PASSWORD_BCRYPT),
                'role' => 'admin'
            ],
            [
                'full_name' => 'John Customer',
                'email' => 'customer@test.com',
                'password' => password_hash('CustomerPass123', PASSWORD_BCRYPT),
                'role' => 'customer'
            ],
            [
                'full_name' => 'Mike Staff',
                'email' => 'staff@test.com',
                'password' => password_hash('StaffPass123', PASSWORD_BCRYPT),
                'role' => 'staff'
            ],
            [
                'full_name' => 'Bob Broker',
                'email' => 'broker@test.com',
                'password' => password_hash('BrokerPass123', PASSWORD_BCRYPT),
                'role' => 'broker'
            ]
        ];
        
        // Insert users
        $inserted_count = 0;
        foreach ($test_users as $user) {
            $stmt = $conn->prepare("INSERT INTO users (full_name, email, password, role, status) VALUES (?, ?, ?, ?, 'active')");
            $stmt->bind_param(
                'ssss',
                $user['full_name'],
                $user['email'],
                $user['password'],
                $user['role']
            );
            
            if ($stmt->execute()) {
                $inserted_count++;
                $setup_messages[] = "✓ Created {$user['role']} account: {$user['email']}";
            } else {
                $setup_messages[] = "❌ Failed to create {$user['email']}: " . $stmt->error;
            }
            $stmt->close();
        }
        
        if ($inserted_count === count($test_users)) {
            $setup_messages[] = "\n✅ All test users created successfully!";
        }
    }
} catch (Exception $e) {
    $setup_messages[] = "❌ Error: " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Zemaraim - Database Setup</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; display: flex; align-items: center; justify-content: center; padding: 20px; }
        .container { background: white; border-radius: 10px; box-shadow: 0 10px 40px rgba(0,0,0,0.2); padding: 40px; max-width: 600px; width: 100%; }
        h1 { color: #667eea; margin-bottom: 10px; font-size: 28px; }
        .subtitle { color: #999; margin-bottom: 30px; font-size: 14px; }
        .messages { background: #f8f9fa; padding: 20px; border-radius: 5px; margin-bottom: 20px; }
        .message { padding: 8px; margin: 5px 0; font-size: 14px; border-radius: 3px; }
        .success { color: #155724; background: #d4edda; border: 1px solid #c3e6cb; }
        .error { color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; }
        .info { color: #004085; background: #d1ecf1; border: 1px solid #bee5eb; }
        .credentials { background: #f0f4ff; border: 2px solid #667eea; padding: 20px; border-radius: 5px; margin-top: 20px; }
        .credentials h3 { color: #667eea; margin-bottom: 15px; font-size: 16px; }
        .cred-item { margin: 10px 0; padding: 10px; background: white; border-radius: 3px; font-size: 13px; font-family: monospace; }
        .cred-item strong { color: #333; }
        .action-buttons { display: flex; gap: 10px; margin-top: 20px; }
        .btn { flex: 1; padding: 12px; border: none; border-radius: 5px; cursor: pointer; font-weight: 600; text-align: center; text-decoration: none; display: inline-block; transition: all 0.3s; }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #764ba2; }
        .btn-secondary { background: #999; color: white; }
        .btn-secondary:hover { background: #777; }
    </style>
</head>
<body>

<div class="container">
    <h1>🏠 Zemaraim Database Setup</h1>
    <p class="subtitle">Initialize test users for development</p>
    
    <div class="messages">
        <?php foreach ($setup_messages as $msg): ?>
            <?php 
                if (strpos($msg, '✓') === 0 || strpos($msg, '✅') === 0) {
                    $class = 'success';
                } elseif (strpos($msg, '❌') === 0) {
                    $class = 'error';
                } else {
                    $class = 'info';
                }
            ?>
            <div class="message <?php echo $class; ?>">
                <?php echo htmlspecialchars($msg); ?>
            </div>
        <?php endforeach; ?>
    </div>
    
    <div class="credentials">
        <h3>📊 Test User Credentials</h3>
        
        <div class="cred-item">
            <strong>Admin:</strong><br>
            Email: admin@test.com | Password: AdminPass123
        </div>
        
        <div class="cred-item">
            <strong>Customer:</strong><br>
            Email: customer@test.com | Password: CustomerPass123
        </div>
        
        <div class="cred-item">
            <strong>Staff:</strong><br>
            Email: staff@test.com | Password: StaffPass123
        </div>
        
        <div class="cred-item">
            <strong>Broker:</strong><br>
            Email: broker@test.com | Password: BrokerPass123
        </div>
    </div>
    
    <div class="action-buttons">
        <a href="admin-login.php" class="btn btn-primary">🔐 Go to Admin Login</a>
        <a href="login.php" class="btn btn-secondary">👤 Go to Customer Login</a>
    </div>
</div>

</body>
</html>
