<?php
/**
 * Zemaraim - Session & Database Fix Verification
 * Verifies all fixes have been applied correctly
 */

require_once 'config.php';

$checks = [];

// Check 1: Session is properly initialized
$checks['Session Started'] = [
    'status' => session_status() === PHP_SESSION_ACTIVE ? 'PASS' : 'FAIL',
    'message' => session_status() === PHP_SESSION_ACTIVE ? 
        'Session is properly initialized' : 
        'Session failed to initialize',
    'details' => 'Session ID: ' . (session_id() ?: 'None')
];

// Check 2: Users table exists
$users_table_exists = false;
$result = $conn->query("SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA = '" . DB_NAME . "' AND TABLE_NAME = 'users'");
if ($result) {
    $row = $result->fetch_assoc();
    $users_table_exists = $row['COUNT(*)'] == 1;
}

$checks['Users Table'] = [
    'status' => $users_table_exists ? 'PASS' : 'FAIL',
    'message' => $users_table_exists ? 
        'Users table exists in database' : 
        'Users table NOT found - run setup-users.php first',
    'details' => 'Database: ' . DB_NAME
];

// Check 3: Count test users
$user_count = 0;
if ($users_table_exists) {
    $result = $conn->query("SELECT COUNT(*) as count FROM users");
    if ($result) {
        $row = $result->fetch_assoc();
        $user_count = $row['count'];
    }
}

$checks['Test Users'] = [
    'status' => $user_count >= 5 ? 'PASS' : 'WARN',
    'message' => $user_count >= 5 ? 
        "All 5 test users created ($user_count found)" : 
        "Test users not created yet ($user_count found) - run setup-users.php",
    'details' => 'Expected: 5, Found: ' . $user_count
];

// Check 4: Admin user exists
$admin_exists = false;
if ($users_table_exists) {
    $result = $conn->query("SELECT COUNT(*) as count FROM users WHERE role = 'admin' AND email = 'admin@test.com'");
    if ($result) {
        $row = $result->fetch_assoc();
        $admin_exists = $row['count'] > 0;
    }
}

$checks['Admin Account'] = [
    'status' => $admin_exists ? 'PASS' : 'WARN',
    'message' => $admin_exists ? 
        'Admin test account exists' : 
        'Admin account not found - run setup-users.php',
    'details' => 'Email: admin@test.com'
];

// Check 5: Config file loads properly
$config_error = null;
$checks['Config File'] = [
    'status' => $config_error === null ? 'PASS' : 'FAIL',
    'message' => 'Config file loaded successfully',
    'details' => 'Location: config.php'
];

// Check 6: Database connection
$checks['Database Connection'] = [
    'status' => !$conn->connect_error ? 'PASS' : 'FAIL',
    'message' => !$conn->connect_error ? 
        'Connected to MySQL successfully' : 
        'Database connection error: ' . $conn->connect_error,
    'details' => 'Host: ' . DB_HOST . ', Database: ' . DB_NAME
];

// Check 7: Test queries work
$query_ok = false;
try {
    $result = $conn->query("SELECT 1");
    $query_ok = $result !== false;
} catch (Exception $e) {
    $query_ok = false;
}

$checks['Query Execution'] = [
    'status' => $query_ok ? 'PASS' : 'FAIL',
    'message' => $query_ok ? 
        'Database queries execute properly' : 
        'Query execution failed',
    'details' => 'Test query: SELECT 1'
];

// Determine overall status
$pass_count = 0;
$warn_count = 0;
$fail_count = 0;
foreach ($checks as $check) {
    if ($check['status'] === 'PASS') $pass_count++;
    elseif ($check['status'] === 'WARN') $warn_count++;
    else $fail_count++;
}

$overall_status = $fail_count > 0 ? 'FAILED' : ($warn_count > 0 ? 'WARNING' : 'SUCCESS');

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Zemaraim - System Verification</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background: #f5f7fa; padding: 20px; }
        .container { max-width: 800px; margin: 0 auto; }
        .header { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.08); margin-bottom: 20px; }
        .header h1 { color: #333; margin-bottom: 10px; font-size: 28px; }
        .status-badge { display: inline-block; padding: 8px 16px; border-radius: 20px; font-weight: 600; font-size: 14px; }
        .status-success { background: #d4edda; color: #155724; }
        .status-warning { background: #fff3cd; color: #856404; }
        .status-failed { background: #f8d7da; color: #721c24; }
        .summary { display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-top: 20px; }
        .summary-item { background: #f8f9fa; padding: 15px; border-radius: 5px; border-left: 3px solid #667eea; }
        .summary-item .number { font-size: 24px; font-weight: 700; color: #667eea; }
        .summary-item .label { font-size: 12px; color: #999; text-transform: uppercase; margin-top: 5px; }
        .checks { background: white; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.08); padding: 20px; margin-bottom: 20px; }
        .check-item { padding: 15px; border-bottom: 1px solid #e0e0e0; display: flex; align-items: flex-start; gap: 15px; }
        .check-item:last-child { border-bottom: none; }
        .check-status { min-width: 80px; }
        .status-pass { background: #d4edda; color: #155724; padding: 6px 12px; border-radius: 4px; text-align: center; font-weight: 600; font-size: 12px; }
        .status-warn { background: #fff3cd; color: #856404; padding: 6px 12px; border-radius: 4px; text-align: center; font-weight: 600; font-size: 12px; }
        .status-fail { background: #f8d7da; color: #721c24; padding: 6px 12px; border-radius: 4px; text-align: center; font-weight: 600; font-size: 12px; }
        .check-content { flex: 1; }
        .check-content h3 { color: #333; font-size: 16px; margin-bottom: 5px; }
        .check-content p { color: #666; font-size: 14px; margin-bottom: 3px; }
        .check-content .details { color: #999; font-size: 12px; font-family: monospace; }
        .actions { background: white; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.08); padding: 20px; display: flex; gap: 10px; }
        .btn { padding: 12px 24px; border: none; border-radius: 5px; cursor: pointer; font-weight: 600; text-decoration: none; display: inline-block; transition: all 0.3s; }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #764ba2; }
        .btn-secondary { background: #999; color: white; }
        .btn-secondary:hover { background: #777; }
        .note { background: #f0f4ff; border-left: 3px solid #667eea; padding: 15px; border-radius: 5px; margin-bottom: 20px; font-size: 13px; color: #333; }
        .note strong { color: #667eea; }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <h1>🔍 Zemaraim System Verification</h1>
        <p style="color: #999; margin-top: 10px;">Verify all fixes and configuration</p>
        
        <div style="margin-top: 15px;">
            <span class="status-badge status-<?php echo strtolower($overall_status === 'SUCCESS' ? 'success' : ($overall_status === 'WARNING' ? 'warning' : 'failed')); ?>">
                Status: <?php echo $overall_status; ?>
            </span>
        </div>
        
        <div class="summary">
            <div class="summary-item">
                <div class="number" style="color: #28a745;"><?php echo $pass_count; ?></div>
                <div class="label">Passed</div>
            </div>
            <div class="summary-item">
                <div class="number" style="color: #ffc107;"><?php echo $warn_count; ?></div>
                <div class="label">Warnings</div>
            </div>
            <div class="summary-item">
                <div class="number" style="color: #dc3545;"><?php echo $fail_count; ?></div>
                <div class="label">Failed</div>
            </div>
        </div>
    </div>
    
    <?php if ($warn_count > 0 || $fail_count > 0): ?>
    <div class="note">
        <strong>⚠️ Action Required:</strong> 
        <?php if ($user_count < 5): ?>
            Please run <strong>setup-users.php</strong> to create test users.
        <?php endif; ?>
    </div>
    <?php endif; ?>
    
    <div class="checks">
        <?php foreach ($checks as $name => $check): ?>
        <div class="check-item">
            <div class="check-status">
                <div class="status-<?php echo strtolower($check['status']); ?>">
                    <?php echo $check['status']; ?>
                </div>
            </div>
            <div class="check-content">
                <h3><?php echo $name; ?></h3>
                <p><?php echo $check['message']; ?></p>
                <div class="details"><?php echo $check['details']; ?></div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    
    <div class="actions">
        <?php if ($user_count < 5): ?>
            <a href="setup-users.php" class="btn btn-primary">📊 Create Test Users</a>
        <?php else: ?>
            <a href="admin-login.php" class="btn btn-primary">🔐 Admin Login</a>
            <a href="login.php" class="btn btn-secondary">👤 Customer Login</a>
        <?php endif; ?>
    </div>
</div>

</body>
</html>
